
// LED Matrix Simulator for 96x96 (P10 mono white) - browser preview
// Renders text into a 96x96 bitmap via offscreen canvas then maps to LEDs.
(function(){
  const LED_COLS = 96, LED_ROWS = 96;

  function clamp(v,min,max){ return Math.max(min, Math.min(max,v)); }

  function makeMatrix(){
    const m = new Uint8Array(LED_COLS*LED_ROWS);
    m.fill(0);
    return m;
  }
  function setPixel(m,x,y,on){
    if(x<0||y<0||x>=LED_COLS||y>=LED_ROWS) return;
    m[y*LED_COLS+x]= on?1:0;
  }
  function getPixel(m,x,y){
    if(x<0||y<0||x>=LED_COLS||y>=LED_ROWS) return 0;
    return m[y*LED_COLS+x];
  }
  function clear(m){ m.fill(0); }

  // Simple erosion to thin overly thick glyphs
  function erode(m, passes=1){
    for(let p=0;p<passes;p++){
      const copy = new Uint8Array(m);
      for(let y=1;y<LED_ROWS-1;y++){
        for(let x=1;x<LED_COLS-1;x++){
          const i=y*LED_COLS+x;
          if(!copy[i]) continue;
          let n=0;
          n += copy[i-1]+copy[i+1]+copy[i-LED_COLS]+copy[i+LED_COLS];
          n += copy[i-LED_COLS-1]+copy[i-LED_COLS+1]+copy[i+LED_COLS-1]+copy[i+LED_COLS+1];
          if(n<=2) m[i]=0;
        }
      }
    }
  }

  const off = document.createElement('canvas');
  off.width = LED_COLS;
  off.height = LED_ROWS;
  const offCtx = off.getContext('2d', { willReadFrequently:true });

  function drawTextToMatrix(m, text, x, y, font, align='left', baseline='top', threshold=180){
    offCtx.clearRect(0,0,LED_COLS,LED_ROWS);
    offCtx.fillStyle = '#000';
    offCtx.fillRect(0,0,LED_COLS,LED_ROWS);
    offCtx.fillStyle = '#fff';
    offCtx.font = font;
    offCtx.textAlign = align;
    offCtx.textBaseline = baseline;
    offCtx.fillText(text, x, y);
    const img = offCtx.getImageData(0,0,LED_COLS,LED_ROWS).data;
    for(let yy=0;yy<LED_ROWS;yy++){
      for(let xx=0;xx<LED_COLS;xx++){
        const idx = (yy*LED_COLS+xx)*4;
        const a = img[idx+3]; // alpha
        const r = img[idx]; // grayscale enough
        const val = (a>0)? r : 0;
        if(val >= threshold) setPixel(m,xx,yy,1);
      }
    }
  }

  function measureTextWidth(font, text){
    offCtx.font = font;
    return offCtx.measureText(text).width;
  }

  function fitFontSingleLine(text, family, weight, maxPx, minPx, maxWidth, maxHeight){
    // iterate down until fits width and height
    for(let px=maxPx; px>=minPx; px--){
      const font = `${weight} ${px}px ${family}`;
      const w = measureTextWidth(font, text);
      const h = px; // rough
      if(w <= maxWidth && h <= maxHeight) return {px, font};
    }
    return {px:minPx, font:`${weight} ${minPx}px ${family}`};
  }

  // Regions per your spec (0-based rows)
  const OFFER_REGIONS = {
    // 0-based inclusive ranges for 96x96 respecting the agreed line map:
    // 1 border, 2-5 space, 6-12 subtitulo, 13-16 space, 17-30 titulo, 31-35 space,
    // 36-78 preço, 79-83 space, 84-91 descrição, 92-95 space, 96 border
    subtitle: {y0:5,  y1:11},   // 6..12 (1-based)
    title:    {y0:16, y1:29},   // 17..30
    price:    {y0:35, y1:77},   // 36..78
    desc:     {y0:83, y1:90},   // 84..91
    borderTop:0,
    borderBottom:95
  };

  function formatPrice(price_cents){
    const reais = Math.floor(price_cents/100);
    const cent = Math.abs(price_cents%100);
    return {reais:String(reais), cent:String(cent).padStart(2,'0')};
  }

  function drawBorder(m, t, speed=5){
    // dashed 3 on / 2 off running clockwise, rounded-ish corners
    const offset = Math.floor(t*speed) % 5;
    let idx=0;

    function dash(i){ return ((i+offset)%5) < 3; }

    // top row y=0, x 0..95
    for(let x=0;x<LED_COLS;x++,idx++){
      if(dash(idx) && !(x<1 && true) && !(x>94 && true)) setPixel(m,x,0,1);
    }
    // right col x=95, y 1..94
    for(let y=1;y<LED_ROWS-1;y++,idx++){
      if(dash(idx)) setPixel(m,95,y,1);
    }
    // bottom y=95, x 95..0
    for(let x=95;x>=0;x--,idx++){
      if(dash(idx)) setPixel(m,x,95,1);
    }
    // left x=0, y 94..1
    for(let y=94;y>=1;y--,idx++){
      if(dash(idx)) setPixel(m,0,y,1);
    }

    // soften corners (rounded impression)
    setPixel(m,0,0,0); setPixel(m,1,0,0); setPixel(m,0,1,0);
    setPixel(m,95,0,0); setPixel(m,94,0,0); setPixel(m,95,1,0);
    setPixel(m,0,95,0); setPixel(m,1,95,0); setPixel(m,0,94,0);
    setPixel(m,95,95,0); setPixel(m,94,95,0); setPixel(m,95,94,0);
  }

  
function renderOffer(m, offer, cfg, t){
    const family = (cfg && cfg.fontFamily) ? cfg.fontFamily : 'Arial';
    const weight = (cfg && cfg.fontWeight) ? cfg.fontWeight : '800';
    const thinPasses = (cfg && cfg.thinPasses!=null) ? cfg.thinPasses : 1;
    const threshold = (cfg && cfg.threshold!=null) ? cfg.threshold : 205;

    const reg = OFFER_REGIONS;

    // texts
    const subtitlePos = (offer.subtitle_position === 'below') ? 'below' : 'above';
    const subtitleText = (offer.subtitle || '').toString();
    const titleText = (offer.title || '').toString();
    const descText = (offer.description || '').toString();
    const unitText = (offer.unit || '').toString().trim();

    // helper to render single line into a box (center vertically)
    function renderBoxText(text, box, maxPx, minPx, align='center'){
      const w = box.x1 - box.x0 + 1;
      const h = box.y1 - box.y0 + 1;
      const fit = fitFontSingleLine(text, family, weight, maxPx, minPx, w, h);
      const x = (align==='left') ? box.x0 : (align==='right') ? box.x1 : Math.floor((box.x0+box.x1)/2);
      const y = box.y0 + Math.floor(h/2);
      const a = (align==='left')?'left':(align==='right')?'right':'center';
      drawTextToMatrix(m, text, x, y, fit.font, a, 'middle', threshold);
      return fit;
    }

    // Decide where subtitle goes
    const subtitleBox = {x0:2, x1:93, y0:reg.subtitle.y0, y1:reg.subtitle.y1};
    const titleBox    = {x0:2, x1:93, y0:reg.title.y0,    y1:reg.title.y1};

    if(subtitlePos==='above'){
      if(subtitleText.trim().length) renderBoxText(subtitleText.toUpperCase(), subtitleBox, 22, 10, 'center');
      if(titleText.trim().length)    renderBoxText(titleText.toUpperCase(),    titleBox,    26, 12, 'center');
    } else {
      if(titleText.trim().length)    renderBoxText(titleText.toUpperCase(),    titleBox,    26, 12, 'center');
      if(subtitleText.trim().length) renderBoxText(subtitleText.toUpperCase(), subtitleBox, 22, 10, 'center');
    }

    // PRICE
    const priceBox = {x0:2, x1:93, y0:reg.price.y0, y1:reg.price.y1};
    const priceH = priceBox.y1-priceBox.y0+1;

    const p = splitPrice(offer.price_cents || 0);
    const reais = p.reais;
    const cent  = p.cent;

    const blinkEnabled = cfg && cfg.price_blink ? 1 : 0;
    const blinkOn = !blinkEnabled || (Math.floor(t*6)%2===0); // 3 flashes-ish per second

    if(blinkOn){
      if(unitText.length){
        // Real (big) at left spanning full height
        const reaisBox = {x0:2, x1:58, y0:priceBox.y0, y1:priceBox.y1}; // left area
        renderBoxText(reais, reaisBox, 80, 24, 'right');

        // Cents area at right top (36..63 => y 35..62)
        const centsBox = {x0:60, x1:93, y0:35, y1:62};
        // include comma before cents
        renderBoxText(','+cent, centsBox, 42, 14, 'left');

        // separator line at row 68 (1-based) => y=67
        const lineY = 67;
        for(let x=60;x<=93;x++) setPixel(m,x,lineY,1);

        // unit at bottom right (72..78 => y 71..77)
        const unitBox = {x0:60, x1:93, y0:71, y1:77};
        renderBoxText(unitText.toUpperCase(), unitBox, 22, 8, 'center');
      } else {
        // no unit -> full price big
        const full = `${reais},${cent}`;
        renderBoxText(full, priceBox, 86, 20, 'center');
      }
    }

    // DESCRIPTION (scroll if overflow)
    if(descText.trim().length){
      const box = {x0:2, x1:93, y0:reg.desc.y0, y1:reg.desc.y1};
      const w = box.x1-box.x0+1;
      const h = box.y1-box.y0+1;

      // choose font size that fits height; width will scroll if needed
      const fit = fitFontSingleLine(descText.toUpperCase(), family, '700', 18, 8, 9999, h);
      const font = fit.font;

      const textW = measureTextWidth(font, descText.toUpperCase());
      if(textW > w){
        const speed = (cfg && cfg.scrollSpeed) ? cfg.scrollSpeed : 12; // px/sec
        const span = textW + w + 6;
        const offset = (t*speed) % span;
        const x = box.x1 - offset;
        const y = box.y0 + Math.floor(h/2);
        drawTextToMatrix(m, descText.toUpperCase(), x, y, font, 'left', 'middle', threshold);
      } else {
        renderBoxText(descText.toUpperCase(), box, 18, 8, 'center');
      }
    }

    // thin to avoid closing counters
    erode(m, thinPasses);
  }


  function renderMessage(m, message, cfg, t){
    const family = (cfg && cfg.fontFamily) ? cfg.fontFamily : 'Arial';
    const threshold = (cfg && cfg.threshold!=null) ? cfg.threshold : 200;
    const thinPasses = (cfg && cfg.thinPasses!=null) ? cfg.thinPasses : 1;

    let lines = [];
    try { lines = JSON.parse(message.lines_json||'[]'); } catch(e){ lines=[]; }
    // Each line has: text, size, align, xoff, yoff
    for(const ln of lines){
      const text = ln.text || '';
      if(!text) continue;
      const px = clamp(parseInt(ln.size||18,10), 8, 64);
      const font = `700 ${px}px ${family}`;
      const align = ln.align || 'center';
      const y = clamp(parseInt(ln.y||48,10), 0, 95);
      const x = clamp(parseInt(ln.x||48,10), 0, 95);
      drawTextToMatrix(m, text, x, y, font, align, 'middle', threshold);
    }
    erode(m, thinPasses);
  }

  function drawToCanvas(m, canvas){
    const ctx = canvas.getContext('2d');
    const scale = canvas.width / LED_COLS;
    ctx.fillStyle = '#000';
    ctx.fillRect(0,0,canvas.width,canvas.height);
    for(let y=0;y<LED_ROWS;y++){
      for(let x=0;x<LED_COLS;x++){
        const on = m[y*LED_COLS+x]===1;
        const cx = x*scale + scale/2;
        const cy = y*scale + scale/2;
        const r = scale*0.38;
        ctx.beginPath();
        ctx.arc(cx, cy, r, 0, Math.PI*2);
        ctx.fillStyle = on ? '#f8f8f8' : '#1a1a1a';
        ctx.fill();
      }
    }
  }

  window.LED_SIM = {
    LED_COLS, LED_ROWS,
    makeMatrix, clear, setPixel, getPixel,
    renderOffer, renderMessage,
    drawBorder,
    drawToCanvas
  };
})();
