<?php
// panel_preview.php
// Preview do painel 96x96 com simulação de LEDs (canvas) + efeitos/transições/borda/piscar preço
// Baseado no preview de ofertas que estava funcionando (sem led_sim.js externo)

require_once 'header.php';
require_once __DIR__ . '/../api/db.php';

$id = intval($_GET['id'] ?? 0);
if ($id <= 0) {
    echo "Painel inválido.";
    require 'footer.php';
    exit;
}

// Busca painel + cliente/loja
$stmt = $mysqli->prepare("SELECT p.id, p.name, p.width, p.height, p.brightness, p.panel_key,
                                 s.name AS store_name,
                                 c.id AS client_id, c.name AS client_name
                          FROM panels p
                          JOIN stores s ON s.id=p.store_id
                          JOIN clients c ON c.id=s.client_id
                          WHERE p.id=?");
$stmt->bind_param('i', $id);
$stmt->execute();
$panel = $stmt->get_result()->fetch_assoc();
if (!$panel) {
    echo "Painel não encontrado.";
    require 'footer.php';
    exit;
}

// Permissão: Admin Master ou Admin do Cliente
if (!is_admin_master() && $panel['client_id'] != $user['client_id']) {
    echo "Acesso negado.";
    require 'footer.php';
    exit;
}

// Configurações do painel (liga/desliga etc.)
$resCfg = $mysqli->query("SELECT power,use_schedule,on_time,off_time FROM panel_settings WHERE panel_id=".(int)$id);
$cfg = $resCfg ? $resCfg->fetch_assoc() : null;
if (!$cfg) {
    $cfg = ['power'=>1,'use_schedule'=>0,'on_time'=>'08:00','off_time'=>'22:00'];
}

// Config global (se existir a tabela)
$global = [
  'transition_effect' => 'fade',
  'duration_sec'      => 8,
  'transition_sec'    => 2,
  'border_animated'   => 1,
  'border_speed'      => 5,
  'price_blink'       => 1,
];
$hasGlobal = false;
$resG = $mysqli->query("SHOW TABLES LIKE 'global_settings'");
if ($resG && $resG->num_rows > 0) {
    $hasGlobal = true;
    $rg = $mysqli->query("SELECT * FROM global_settings WHERE id=1");
    if ($rg && ($row = $rg->fetch_assoc())) {
        $global['transition_effect'] = $row['transition_effect'] ?? $global['transition_effect'];
        $global['duration_sec']      = intval($row['duration_sec'] ?? $global['duration_sec']);
        $global['transition_sec']    = intval($row['transition_sec'] ?? $global['transition_sec']);
        $global['border_animated']   = intval($row['border_animated'] ?? $global['border_animated']);
        $global['border_speed']      = intval($row['border_speed'] ?? $global['border_speed']);
        $global['price_blink']       = intval($row['price_blink'] ?? $global['price_blink']);
    }
}

// Ofertas associadas ao painel
$sqlOff = "SELECT o.*
           FROM panel_offers po
           JOIN offers o ON o.id = po.offer_id
           WHERE po.panel_id = ".(int)$id." AND po.enabled = 1 AND o.enabled = 1
           ORDER BY po.position ASC, o.id ASC";
$resOff = $mysqli->query($sqlOff);
$offers = [];
if ($resOff) {
    while ($o = $resOff->fetch_assoc()) $offers[] = $o;
}

// ====== FONTES AUTOMÁTICAS (admin/fonts/*.ttf / *.otf) ======
$fontsDir = __DIR__ . '/fonts';
$fontFamilies = [];
if (is_dir($fontsDir)) {
    $files = scandir($fontsDir);
    foreach ($files as $f) {
        if (preg_match('/\.(ttf|otf)$/i', $f)) {
            $ext = pathinfo($f, PATHINFO_EXTENSION);
            $base = pathinfo($f, PATHINFO_FILENAME);
            $family = preg_replace('/\s+/', '_', $base);
            $fontFamilies[] = [
                'file' => $f,
                'family' => $family,
                'ext' => strtolower($ext),
            ];
        }
    }
}
$fontNames = array_map(fn($x)=>$x['family'], $fontFamilies);

?>
<div class="container-fluid">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <div>
      <h3 class="mb-0">Preview do Painel: <?= htmlspecialchars($panel['name']) ?></h3>
      <div class="text-muted">
        <?= htmlspecialchars($panel['client_name']) ?> / <?= htmlspecialchars($panel['store_name']) ?>
      </div>
    </div>
  </div>

  <?php if (!$cfg['power']): ?>
    <div class="alert alert-warning">⚠ Painel configurado como <strong>DESLIGADO</strong> (power=0). No ESP32 ele ficaria apagado.</div>
  <?php endif; ?>

  <?php if (empty($offers)): ?>
    <div class="alert alert-info">Nenhuma oferta associada a este painel. Adicione em <strong>Painéis → Ofertas do Painel</strong>.</div>
  <?php endif; ?>

  <div class="row g-3">
    <div class="col-lg-8">
      <div class="mb-2">
        <label class="form-label mb-1"><strong>Fonte usada no painel:</strong></label>
        <select id="fontSelector" class="form-select form-select-sm" style="max-width:320px;">
          <?php if (!empty($fontFamilies)): ?>
            <?php foreach ($fontFamilies as $ff): ?>
              <option value="<?= htmlspecialchars($ff['family']) ?>"><?= htmlspecialchars($ff['family']) ?></option>
            <?php endforeach; ?>
          <?php else: ?>
            <option value="">(Padrão do sistema)</option>
          <?php endif; ?>
        </select>
        <div class="small text-muted mt-1">
          Coloque arquivos <code>.ttf</code> / <code>.otf</code> em <code>admin/fonts</code> e recarregue.
        </div>
      </div>

      <div class="d-flex gap-2 mb-3">
        <button class="btn btn-sm btn-outline-primary" id="btnStart">▶ Iniciar pré-visualização</button>
        <button class="btn btn-sm btn-outline-secondary" id="btnStop">⏹ Parar</button>
      </div>

      <style>
        /* Fontes auto */
        <?php foreach ($fontFamilies as $ff): ?>
        @font-face {
          font-family: '<?= $ff['family']; ?>';
          src: url('fonts/<?= htmlspecialchars($ff['file'], ENT_QUOTES); ?>') format('truetype');
          font-weight: normal;
          font-style: normal;
        }
        <?php endforeach; ?>

        #panel-wrapper {
          width: 600px;
          height: 600px;
          background: #000;
          display: flex;
          align-items: center;
          justify-content: center;
          border-radius: 12px;
          box-shadow: 0 0 30px rgba(0,0,0,0.75);
          overflow: hidden;
          border: 1px solid rgba(255,255,255,0.05);
        }
        #panel-canvas {
          width: 600px;
          height: 600px;
          display: block;
          background: #000;
        }
      </style>

      <div id="panel-wrapper">
        <canvas id="panel-canvas" width="600" height="600"></canvas>
      </div>

      <div class="small text-muted mt-2">
        Dica: este preview é uma simulação. O ESP32 vai renderizar com a biblioteca do painel (HUB12), então o “look” fica muito próximo, mas pode variar levemente.
      </div>
    </div>

    <div class="col-lg-4">
      <div class="card">
        <div class="card-header"><strong>Teste rápido (só no preview)</strong></div>
        <div class="card-body">

          <div class="mb-3">
            <label class="form-label">Efeito</label>
            <select id="cfgEffect" class="form-select form-select-sm">
              <option value="fade">Fade</option>
              <option value="left">Entrada da direita → esquerda</option>
              <option value="right">Entrada da esquerda → direita</option>
              <option value="up">Entrada de baixo → cima</option>
              <option value="down">Entrada de cima → baixo</option>
            </select>
          </div>

          <div class="mb-3">
            <label class="form-label">Tempo da oferta (5–15s)</label>
            <input id="cfgDuration" type="number" min="5" max="15" class="form-control form-control-sm">
          </div>

          <div class="mb-3">
            <label class="form-label">Tempo de transição (1–5s)</label>
            <input id="cfgTransition" type="number" min="1" max="5" class="form-control form-control-sm">
          </div>

          <div class="form-check mb-2">
            <input class="form-check-input" type="checkbox" id="cfgBorder">
            <label class="form-check-label" for="cfgBorder">Borda animada (3 sim / 2 não)</label>
          </div>

          <div class="mb-3">
            <label class="form-label">Velocidade da borda</label>
            <input id="cfgBorderSpeed" type="number" min="1" max="20" class="form-control form-control-sm">
          </div>

          <div class="form-check mb-3">
            <input class="form-check-input" type="checkbox" id="cfgBlink">
            <label class="form-check-label" for="cfgBlink">Piscar preço</label>
          </div>

          <div class="mb-3">
            <label class="form-label">Afinar (0–2)</label>
            <input id="cfgThin" type="number" min="0" max="2" class="form-control form-control-sm" value="1">
            <div class="small text-muted mt-1">Ajuda a não “fechar” detalhes quando a fonte fica grossa.</div>
          </div>

          <div class="mb-3">
            <label class="form-label">Threshold (150–230)</label>
            <input id="cfgThreshold" type="number" min="150" max="230" class="form-control form-control-sm" value="205">
          </div>

          <button class="btn btn-sm btn-outline-success w-100" id="btnApply">Aplicar no preview</button>

        </div>
      </div>

      <div class="alert alert-secondary mt-3 mb-0">
        <small>
          <strong>Obs:</strong> Essas configs aqui são só para testar no preview.
          Depois a gente liga isso no “Global → Painel → Oferta/Mensagem”.
        </small>
      </div>

    </div>
  </div>
</div>

<script>
// ================== DADOS PHP -> JS ==================
const offers = <?php
$out = [];
foreach ($offers as $o) {
    $out[] = [
        'id' => (int)$o['id'],
        'name' => $o['name'] ?? '',
        'header' => $o['header'] ?? '',
        'title' => $o['title'] ?? '',
        'subtitle' => $o['subtitle'] ?? '',
        'price_cents' => (int)($o['price_cents'] ?? 0),
        'unit' => $o['unit'] ?? '',
        'aux'  => $o['aux'] ?? '',
        'footer' => $o['footer'] ?? '',
        'duration_sec' => (int)($o['duration_sec'] ?? 8),
    ];
}
echo json_encode($out, JSON_UNESCAPED_UNICODE);
?>;

const globalDefaults = <?php echo json_encode($global, JSON_UNESCAPED_UNICODE); ?>;

const availableFonts = <?php echo json_encode($fontNames, JSON_UNESCAPED_UNICODE); ?>;
let selectedFont = (availableFonts.length > 0) ? availableFonts[0] : "";

// ================== UI INIT ==================
const elFont = document.getElementById("fontSelector");
const elEffect = document.getElementById("cfgEffect");
const elDuration = document.getElementById("cfgDuration");
const elTransition = document.getElementById("cfgTransition");
const elBorder = document.getElementById("cfgBorder");
const elBorderSpeed = document.getElementById("cfgBorderSpeed");
const elBlink = document.getElementById("cfgBlink");
const elThin = document.getElementById("cfgThin");
const elThreshold = document.getElementById("cfgThreshold");

function loadUIFromDefaults() {
  elEffect.value = globalDefaults.transition_effect || 'fade';
  elDuration.value = clampInt(globalDefaults.duration_sec || 8, 5, 15);
  elTransition.value = clampInt(globalDefaults.transition_sec || 2, 1, 5);
  elBorder.checked = !!(globalDefaults.border_animated ?? 1);
  elBorderSpeed.value = clampInt(globalDefaults.border_speed || 5, 1, 20);
  elBlink.checked = !!(globalDefaults.price_blink ?? 1);
}
loadUIFromDefaults();

if (elFont) {
  elFont.addEventListener("change", () => {
    selectedFont = elFont.value;
    // não reinicia automaticamente: mantém controle do usuário
  });
}

document.getElementById("btnApply").addEventListener("click", () => {
  previewCfg.effect = elEffect.value || 'fade';
  previewCfg.durationSec = clampInt(elDuration.value, 5, 15);
  previewCfg.transitionSec = clampInt(elTransition.value, 1, 5);
  previewCfg.borderAnimated = !!elBorder.checked;
  previewCfg.borderSpeed = clampInt(elBorderSpeed.value, 1, 20);
  previewCfg.priceBlink = !!elBlink.checked;
  previewCfg.thin = clampInt(elThin.value, 0, 2);
  previewCfg.threshold = clampInt(elThreshold.value, 150, 230);
});

document.getElementById("btnStart").addEventListener("click", startPreview);
document.getElementById("btnStop").addEventListener("click", stopPreview);

function getFont() {
  if (selectedFont && selectedFont.length > 0) return selectedFont + ", sans-serif";
  return "sans-serif";
}

// ================== CANVAS / LED SIM ==================
const LED_COLS = 96;
const LED_ROWS = 96;

const canvas = document.getElementById('panel-canvas');
const ctx = canvas.getContext('2d');

const CANVAS_W = canvas.width;   // 600
const CANVAS_H = canvas.height;  // 600

const STEP_X = CANVAS_W / LED_COLS;
const STEP_Y = CANVAS_H / LED_ROWS;

// OFFSCREEN: desenha “sólido” e converte pra matriz de LEDs
const raw = document.createElement('canvas');
raw.width = LED_COLS;
raw.height = LED_ROWS;
// willReadFrequently para evitar warning e ficar mais rápido
const rawCtx = raw.getContext('2d', { willReadFrequently: true });

let running = false;
let currentIndex = 0;
let timer = null;
let borderTick = 0;

const previewCfg = {
  effect: elEffect.value || 'fade',
  durationSec: clampInt(elDuration.value, 5, 15),
  transitionSec: clampInt(elTransition.value, 1, 5),
  borderAnimated: !!elBorder.checked,
  borderSpeed: clampInt(elBorderSpeed.value, 1, 20),
  priceBlink: !!elBlink.checked,
  thin: clampInt(elThin.value, 0, 2),
  threshold: clampInt(elThreshold.value, 150, 230),
};

// Frame buffers (96x96 boolean)
let frameA = null; // atual
let frameB = null; // próximo

function clearCanvas() {
  ctx.fillStyle = '#000';
  ctx.fillRect(0,0,CANVAS_W,CANVAS_H);
}

// ================== RENDER HELPERS ==================
function clampInt(v, min, max) {
  const n = parseInt(v, 10);
  if (isNaN(n)) return min;
  return Math.max(min, Math.min(max, n));
}

function splitPrice(priceCents) {
  // priceCents ex: 899 -> "8,99"
  const cents = Math.abs(parseInt(priceCents || 0, 10));
  const reais = Math.floor(cents / 100);
  const cent = (cents % 100).toString().padStart(2, '0');
  return { reais: reais.toString(), cents: cent };
}

function textToMatrix(drawFn, threshold=205, thin=1) {
  // desenha em raw 96x96 como branco sólido e converte em matriz bool
  rawCtx.clearRect(0,0,LED_COLS,LED_ROWS);
  rawCtx.fillStyle = "#000";
  rawCtx.fillRect(0,0,LED_COLS,LED_ROWS);

  drawFn(rawCtx);

  const img = rawCtx.getImageData(0,0,LED_COLS,LED_ROWS).data;
  const m = new Array(LED_ROWS);
  for (let y=0; y<LED_ROWS; y++) {
    m[y] = new Array(LED_COLS).fill(false);
    for (let x=0; x<LED_COLS; x++) {
      const i = (y*LED_COLS + x)*4;
      const v = img[i]; // R
      m[y][x] = v >= threshold;
    }
  }

  // thinning simples para não “fechar” detalhes (remove pixels isolados)
  // thin=0 (nada), 1 leve, 2 mais forte
  if (thin > 0) {
    for (let pass=0; pass<thin; pass++) {
      const copy = m.map(row => row.slice());
      for (let y=1; y<LED_ROWS-1; y++) {
        for (let x=1; x<LED_COLS-1; x++) {
          if (!copy[y][x]) continue;
          // conta vizinhos
          let n=0;
          for (let yy=-1; yy<=1; yy++) for (let xx=-1; xx<=1; xx++) {
            if (yy===0 && xx===0) continue;
            if (copy[y+yy][x+xx]) n++;
          }
          // se quase sem vizinhos, apaga (reduz “gordura”)
          if (n <= 1) m[y][x] = false;
        }
      }
    }
  }

  return m;
}

function drawMatrixToLED(matrix, alpha=1.0, xOffset=0, yOffset=0) {
  // desenha os “LEDs” no canvas 600x600
  // fundo preto total, apagados em chumbo quase preto
  clearCanvas();

  const rOn = 255, gOn = 255, bOn = 255;
  const rOff = 30,  gOff = 30,  bOff = 30;

  // tamanho do “LED dip”
  const ledRadius = Math.min(STEP_X, STEP_Y) * 0.42;

  for (let y=0; y<LED_ROWS; y++) {
    for (let x=0; x<LED_COLS; x++) {
      const xx = (x + xOffset) * STEP_X + STEP_X/2;
      const yy = (y + yOffset) * STEP_Y + STEP_Y/2;
      if (xx < 0 || yy < 0 || xx > CANVAS_W || yy > CANVAS_H) continue;

      const on = matrix[y][x];
      const c = on ? `rgba(${rOn},${gOn},${bOn},${alpha})` : `rgb(${rOff},${gOff},${bOff})`;

      ctx.beginPath();
      ctx.fillStyle = c;
      ctx.arc(xx, yy, ledRadius, 0, Math.PI*2);
      ctx.fill();
    }
  }
}

function drawBorderOverlay() {
  if (!previewCfg.borderAnimated) return;

  // borda “correndo”: 3 on / 2 off ao redor
  // offset = borderTick
  const pattern = [1,1,1,0,0];
  const offset = borderTick % pattern.length;

  const setLED = (x,y,on) => {
    // desenha em cima (mais forte)
    const xx = x * STEP_X + STEP_X/2;
    const yy = y * STEP_Y + STEP_Y/2;
    const ledRadius = Math.min(STEP_X, STEP_Y) * 0.42;
    ctx.beginPath();
    ctx.fillStyle = on ? `rgb(255,255,255)` : `rgb(30,30,30)`;
    ctx.arc(xx, yy, ledRadius, 0, Math.PI*2);
    ctx.fill();
  };

  // cantos arredondados (pequeno “raio”)
  const R = 3; // raio em LEDs

  // topo
  let idx=0;
  for (let x=0; x<LED_COLS; x++) {
    const inCorner = (x < R) || (x >= LED_COLS-R);
    if (inCorner) { idx++; continue; }
    const on = pattern[(idx+offset)%pattern.length]===1;
    setLED(x,0,on);
    idx++;
  }
  // base
  idx=0;
  for (let x=0; x<LED_COLS; x++) {
    const inCorner = (x < R) || (x >= LED_COLS-R);
    if (inCorner) { idx++; continue; }
    const on = pattern[(idx+offset)%pattern.length]===1;
    setLED(x,LED_ROWS-1,on);
    idx++;
  }
  // esquerda
  idx=0;
  for (let y=0; y<LED_ROWS; y++) {
    const inCorner = (y < R) || (y >= LED_ROWS-R);
    if (inCorner) { idx++; continue; }
    const on = pattern[(idx+offset)%pattern.length]===1;
    setLED(0,y,on);
    idx++;
  }
  // direita
  idx=0;
  for (let y=0; y<LED_ROWS; y++) {
    const inCorner = (y < R) || (y >= LED_ROWS-R);
    if (inCorner) { idx++; continue; }
    const on = pattern[(idx+offset)%pattern.length]===1;
    setLED(LED_COLS-1,y,on);
    idx++;
  }

  // cantos arredondados simples (diagonal)
  for (let i=0;i<R;i++){
    const on = true;
    // sup esq
    setLED(i,0,on); setLED(0,i,on);
    // sup dir
    setLED(LED_COLS-1-i,0,on); setLED(LED_COLS-1,i,on);
    // inf esq
    setLED(i,LED_ROWS-1,on); setLED(0,LED_ROWS-1-i,on);
    // inf dir
    setLED(LED_COLS-1-i,LED_ROWS-1,on); setLED(LED_COLS-1,LED_ROWS-1-i,on);
  }
}

// ================== LAYOUT DA OFERTA ==================
function drawOfferRaw(ctx96, offer, blinkOn=true) {
  // Grid em linhas (as mesmas regras que você definiu)
  // 1 borda (a borda animada é overlay no canvas principal; aqui focamos no conteúdo)
  // 5-12 subtitulo
  // 17-30 titulo
  // 36-78 preço
  // 84-91 descrição (footer)
  // OBS: aqui é um “auto-fit” simples, sem o pacote completo de todas as regras ainda.
  //      Mas já respeita blocos e faz encaixe.

  ctx96.save();
  ctx96.fillStyle = "#000";
  ctx96.fillRect(0,0,LED_COLS,LED_ROWS);

  ctx96.fillStyle = "#fff";
  ctx96.textBaseline = "top";
  ctx96.textAlign = "center";

  const fontFamily = getFont();

  const header = (offer.header || "").trim();
  const title = (offer.title || "").trim();
  const subtitle = (offer.subtitle || "").trim();
  const footer = (offer.footer || "").trim();
  const unit = (offer.unit || "").trim();

  // Subtítulo (linhas 5..12) => y ~ 4..11
  if (subtitle.length > 0) {
    const boxY = 4, boxH = 8;
    let size = 10;
    while (size > 5) {
      ctx96.font = `${size}px ${fontFamily}`;
      if (ctx96.measureText(subtitle).width <= 90) break;
      size--;
    }
    ctx96.fillText(subtitle, 48, boxY + Math.floor((boxH - size)/2));
  } else if (header.length > 0) {
    const boxY = 4, boxH = 8;
    let size = 10;
    while (size > 5) {
      ctx96.font = `${size}px ${fontFamily}`;
      if (ctx96.measureText(header).width <= 90) break;
      size--;
    }
    ctx96.fillText(header, 48, boxY + Math.floor((boxH - size)/2));
  }

  // Título (linhas 17..30) => y 16..29
  if (title.length > 0) {
    const boxY = 16, boxH = 14;
    let size = 16;
    while (size > 7) {
      ctx96.font = `${size}px ${fontFamily}`;
      if (ctx96.measureText(title).width <= 90) break;
      size--;
    }
    ctx96.fillText(title, 48, boxY + Math.floor((boxH - size)/2));
  }

  // Preço (linhas 36..78) => y 35..77
  const boxPY = 35, boxPH = 43;
  const p = splitPrice(offer.price_cents || 0);
  const reais = p.reais;
  const cents = p.cents;

  // se não tem unidade, centavos ficam grandes também
  const showUnit = unit.length > 0;

  // Reais (grande)
  let sizeReais = 40;
  while (sizeReais > 18) {
    ctx96.font = `bold ${sizeReais}px ${fontFamily}`;
    const w = ctx96.measureText(reais).width;
    if (w <= 56 && sizeReais <= boxPH) break;
    sizeReais--;
  }

  const baseY = boxPY + 3;

  if (blinkOn) {
    ctx96.fillText(reais, 30, baseY); // esquerda (reais)
  }

  // Centavos / Unidade na direita
  if (showUnit) {
    // centavos menores, com linha e unidade abaixo
    let sizeC = 18;
    while (sizeC > 10) {
      ctx96.font = `bold ${sizeC}px ${fontFamily}`;
      if (ctx96.measureText(cents).width <= 26) break;
      sizeC--;
    }

    if (blinkOn) {
      // vírgula + cents
      ctx96.font = `bold ${Math.max(12, sizeC)}px ${fontFamily}`;
      ctx96.textAlign = "left";
      ctx96.fillText("," + cents, 58, baseY + 14);
      ctx96.textAlign = "center";

      // linha separadora
      ctx96.fillRect(58, baseY + 14 + sizeC + 1, 30, 1);

      // unidade menor
      let sizeU = 14;
      while (sizeU > 8) {
        ctx96.font = `bold ${sizeU}px ${fontFamily}`;
        if (ctx96.measureText(unit).width <= 30) break;
        sizeU--;
      }
      ctx96.font = `bold ${sizeU}px ${fontFamily}`;
      ctx96.textAlign = "left";
      ctx96.fillText(unit, 58, baseY + 14 + sizeC + 4);
      ctx96.textAlign = "center";
    }
  } else {
    // sem unidade -> centavos grandes e alinhados
    let sizeC = 26;
    while (sizeC > 14) {
      ctx96.font = `bold ${sizeC}px ${fontFamily}`;
      if (ctx96.measureText("," + cents).width <= 40) break;
      sizeC--;
    }
    if (blinkOn) {
      ctx96.font = `bold ${sizeC}px ${fontFamily}`;
      ctx96.textAlign = "left";
      ctx96.fillText("," + cents, 58, baseY + 10);
      ctx96.textAlign = "center";
    }
  }

  // Footer / descrição (linhas 84..91) => y 83..90
  if (footer.length > 0) {
    const boxY = 83, boxH = 8;
    let size = 9;
    while (size > 6) {
      ctx96.font = `${size}px ${fontFamily}`;
      if (ctx96.measureText(footer).width <= 90) break;
      size--;
    }
    ctx96.font = `${size}px ${fontFamily}`;
    ctx96.fillText(footer, 48, boxY + Math.floor((boxH - size)/2));
  }

  ctx96.restore();
}

function computeFrameForOffer(offer, blinkOn=true) {
  return textToMatrix((c)=>drawOfferRaw(c, offer, blinkOn), previewCfg.threshold, previewCfg.thin);
}

// ================== TRANSIÇÃO ==================
async function transition(fromFrame, toFrame, effect, durationMs, withBorder=true) {
  const start = performance.now();

  return new Promise((resolve) => {
    function step(t) {
      const p = Math.min(1, (t - start) / durationMs);

      let alphaA = 1, alphaB = 1;
      let xOffA=0, yOffA=0, xOffB=0, yOffB=0;

      if (effect === 'fade') {
        alphaA = 1 - p;
        alphaB = p;
      } else if (effect === 'left') {
        // novo entra da direita para esquerda
        xOffA = -p * LED_COLS;
        xOffB = (1 - p) * LED_COLS;
      } else if (effect === 'right') {
        xOffA = p * LED_COLS;
        xOffB = -(1 - p) * LED_COLS;
      } else if (effect === 'up') {
        yOffA = -p * LED_ROWS;
        yOffB = (1 - p) * LED_ROWS;
      } else if (effect === 'down') {
        yOffA = p * LED_ROWS;
        yOffB = -(1 - p) * LED_ROWS;
      }

      // desenha A e B (composição simples)
      clearCanvas();
      if (fromFrame) drawMatrixToLED(fromFrame, alphaA, xOffA, yOffA);
      if (toFrame) {
        // desenha por cima
        // truque: desenha sem limpar desta vez -> precisamos de função que NÃO limpe
        // então: redesenhamos B manualmente em overlay:
        const rOn=255,gOn=255,bOn=255;
        const rOff=30,gOff=30,bOff=30;
        const ledRadius = Math.min(STEP_X, STEP_Y) * 0.42;

        for (let y=0; y<LED_ROWS; y++) {
          for (let x=0; x<LED_COLS; x++) {
            const xx = (x + xOffB) * STEP_X + STEP_X/2;
            const yy = (y + yOffB) * STEP_Y + STEP_Y/2;
            if (xx < 0 || yy < 0 || xx > CANVAS_W || yy > CANVAS_H) continue;

            const on = toFrame[y][x];
            const c = on ? `rgba(${rOn},${gOn},${bOn},${alphaB})` : `rgb(${rOff},${gOff},${bOff})`;
            ctx.beginPath();
            ctx.fillStyle = c;
            ctx.arc(xx, yy, ledRadius, 0, Math.PI*2);
            ctx.fill();
          }
        }
      }

      if (withBorder) drawBorderOverlay();

      if (p < 1 && running) requestAnimationFrame(step);
      else resolve();
    }
    requestAnimationFrame(step);
  });
}

// ================== LOOP PRINCIPAL ==================
function startPreview() {
  if (running) return;
  if (!offers || offers.length === 0) return;

  // aplica UI -> cfg
  previewCfg.effect = elEffect.value || 'fade';
  previewCfg.durationSec = clampInt(elDuration.value, 5, 15);
  previewCfg.transitionSec = clampInt(elTransition.value, 1, 5);
  previewCfg.borderAnimated = !!elBorder.checked;
  previewCfg.borderSpeed = clampInt(elBorderSpeed.value, 1, 20);
  previewCfg.priceBlink = !!elBlink.checked;
  previewCfg.thin = clampInt(elThin.value, 0, 2);
  previewCfg.threshold = clampInt(elThreshold.value, 150, 230);

  selectedFont = elFont ? elFont.value : selectedFont;

  running = true;
  currentIndex = 0;
  borderTick = 0;

  // inicia com o primeiro frame (com blink ON)
  frameA = computeFrameForOffer(offers[currentIndex], true);
  drawMatrixToLED(frameA, 1, 0, 0);
  drawBorderOverlay();

  scheduleNext();
  startBorderTicker();
}

function stopPreview() {
  running = false;
  if (timer) clearTimeout(timer);
  timer = null;
  clearCanvas();
}

function startBorderTicker() {
  // atualiza borda “correndo”
  const interval = Math.max(30, 220 - (previewCfg.borderSpeed*10)); // speed maior => mais rápido
  (function tickBorder(){
    if (!running) return;
    borderTick++;
    // redesenha frame atual com borda atual (sem mudar conteúdo)
    if (frameA) {
      drawMatrixToLED(frameA, 1, 0, 0);
      drawBorderOverlay();
    }
    setTimeout(tickBorder, interval);
  })();
}

function scheduleNext() {
  if (!running) return;
  const offer = offers[currentIndex];

  const offerSec = clampInt(previewCfg.durationSec, 5, 15);
  const transMs = clampInt(previewCfg.transitionSec, 1, 5) * 1000;

  // Piscar preço (3 piscadas no começo)
  let blinkCount = previewCfg.priceBlink ? 3 : 0;
  let blinkOn = true;

  const doBlink = (cb) => {
    if (!running) return;
    if (blinkCount <= 0) { cb(); return; }
    blinkOn = !blinkOn;
    frameA = computeFrameForOffer(offer, blinkOn);
    drawMatrixToLED(frameA, 1, 0, 0);
    drawBorderOverlay();
    blinkCount--;
    setTimeout(()=>doBlink(cb), 180);
  };

  doBlink(async () => {
    // mantém exibindo até trocar
    timer = setTimeout(async () => {
      if (!running) return;

      const nextIndex = (currentIndex + 1) % offers.length;
      const nextOffer = offers[nextIndex];

      // frame atual fixo (blink ON) e frame próximo (blink ON)
      const from = computeFrameForOffer(offer, true);
      const to = computeFrameForOffer(nextOffer, true);

      await transition(from, to, previewCfg.effect, transMs, true);

      currentIndex = nextIndex;
      frameA = to;

      scheduleNext();
    }, offerSec * 1000);
  });
}

// start/stop handlers (buttons)
function startPreviewBtn(){ startPreview(); }
function stopPreviewBtn(){ stopPreview(); }

</script>

<?php require 'footer.php'; ?>
