<?php
require_once 'header.php';
require_once __DIR__ . '/../api/db.php';

$id = intval($_GET['id'] ?? 0);

$message = [
  'id' => 0,
  'client_id' => $user['client_id'],
  'name' => '',
  'enabled' => 1,
  'use_global' => 1,
  'transition_effect' => 'fade',
  'duration_sec' => 8,
  'border_animated' => 1,
  'border_speed' => 5,
  'lines_json' => '[]'
];

if ($id > 0) {
  // segurança: limita pelo client_id do usuário (exceto admin master)
  if (is_admin_master()) {
    $stmt = $mysqli->prepare("SELECT * FROM messages WHERE id=?");
    $stmt->bind_param('i',$id);
  } else {
    $stmt = $mysqli->prepare("SELECT * FROM messages WHERE id=? AND client_id=?");
    $stmt->bind_param('ii',$id,$user['client_id']);
  }
  $stmt->execute();
  $res = $stmt->get_result();
  if ($row = $res->fetch_assoc()) {
    $message = $row;
  } else {
    echo "Mensagem não encontrada.";
    require 'footer.php';
    exit;
  }
}

if ($_SERVER['REQUEST_METHOD']==='POST') {
  $client_id = is_admin_master() ? intval($_POST['client_id'] ?? $user['client_id']) : $user['client_id'];
  $name = $_POST['name'] ?? '';
  $enabled = isset($_POST['enabled']) ? 1 : 0;
  $use_global = isset($_POST['use_global']) ? 1 : 0;

  $transition_effect = $_POST['transition_effect'] ?? 'fade';
  $duration_sec = max(5, min(15, intval($_POST['duration_sec'] ?? 8)));
  $border_animated = isset($_POST['border_animated']) ? 1 : 0;
  $border_speed = max(1, min(10, intval($_POST['border_speed'] ?? 5)));

  // linhas
  $texts = $_POST['line_text'] ?? [];
  $sizes = $_POST['line_size'] ?? [];
  $aligns = $_POST['line_align'] ?? [];
  $ys = $_POST['line_y'] ?? [];
  $offsets = $_POST['line_offset'] ?? [];
  $scrolls = $_POST['line_scroll'] ?? [];
  $scroll_speeds = $_POST['line_scroll_speed'] ?? [];

  $lines = [];
  $count = count($texts);
  for ($i=0;$i<$count;$i++) {
    $t = trim($texts[$i]);
    if ($t === '') continue;
    $lines[] = [
      'text' => $t,
      'size' => max(6, min(48, intval($sizes[$i] ?? 16))),
      'align' => $aligns[$i] ?? 'center',
      'y' => max(0, min(95, intval($ys[$i] ?? 20))),
      'offset' => intval($offsets[$i] ?? 0),
      'scroll' => isset($scrolls[$i]) ? 1 : 0,
      'scroll_speed' => max(1, min(20, intval($scroll_speeds[$i] ?? 8)))
    ];
  }
  $lines_json = json_encode($lines, JSON_UNESCAPED_UNICODE);

  if ($id > 0) {
    $stmt = $mysqli->prepare("UPDATE messages SET client_id=?, name=?, enabled=?, use_global=?, transition_effect=?, duration_sec=?, border_animated=?, border_speed=?, lines_json=? WHERE id=?");
    $stmt->bind_param('isiiisiisi',
      $client_id, $name, $enabled, $use_global,
      $transition_effect, $duration_sec, $border_animated,
      $border_speed, $lines_json, $id
    );
  } else {
    $stmt = $mysqli->prepare("INSERT INTO messages (client_id,name,enabled,use_global,transition_effect,duration_sec,border_animated,border_speed,lines_json) VALUES (?,?,?,?,?,?,?,?,?)");
    $stmt->bind_param('isiiisiis',
      $client_id, $name, $enabled, $use_global,
      $transition_effect, $duration_sec, $border_animated,
      $border_speed, $lines_json
    );
  }
  if ($stmt->execute()) {
    echo '<div class="alert alert-success">Mensagem salva.</div>';
    if ($id == 0) {
      $id = $mysqli->insert_id;
    }
    // recarrega dados
    $message['client_id'] = $client_id;
    $message['name'] = $name;
    $message['enabled'] = $enabled;
    $message['use_global'] = $use_global;
    $message['transition_effect'] = $transition_effect;
    $message['duration_sec'] = $duration_sec;
    $message['border_animated'] = $border_animated;
    $message['border_speed'] = $border_speed;
    $message['lines_json'] = $lines_json;
  } else {
    echo '<div class="alert alert-danger">Erro ao salvar.</div>';
  }
}

$lines = json_decode($message['lines_json'], true) ?: [];
if (empty($lines)) {
  $lines = [[
    'text' => 'NOVA MENSAGEM',
    'size' => 18,
    'align' => 'center',
    'y' => 32,
    'offset' => 0,
    'scroll' => 0,
    'scroll_speed' => 8
  ]];
}

$effects = [
  'fade' => 'Fade',
  'slide_right' => 'Entrada direita',
  'slide_left' => 'Entrada esquerda',
  'slide_top' => 'Entrada cima',
  'slide_bottom' => 'Entrada baixo',
  'zoom' => 'Zoom',
  'dissolve' => 'Dissolve',
];
?>
<h3><?= $id ? 'Editar Mensagem' : 'Nova Mensagem' ?></h3>
<div class="row">
  <div class="col-md-5">
    <form method="post">
      <?php if (is_admin_master()): ?>
      <div class="mb-2">
        <label class="form-label">Cliente</label>
        <select name="client_id" class="form-select form-select-sm">
          <?php
          $cres = $mysqli->query("SELECT id,name FROM clients ORDER BY name");
          while($c = $cres->fetch_assoc()):
          ?>
            <option value="<?= $c['id'] ?>" <?= $c['id']==$message['client_id']?'selected':'' ?>>
              <?= htmlspecialchars($c['name']) ?>
            </option>
          <?php endwhile; ?>
        </select>
      </div>
      <?php endif; ?>

      <div class="mb-2">
        <label class="form-label">Nome interno</label>
        <input type="text" name="name" class="form-control form-control-sm" value="<?= htmlspecialchars($message['name']) ?>" required>
      </div>
      <div class="form-check mb-2">
        <input class="form-check-input" type="checkbox" name="enabled" id="enabled" <?= $message['enabled'] ? 'checked' : '' ?>>
        <label class="form-check-label" for="enabled">Mensagem ativa</label>
      </div>

      <hr>
      <h6>Configuração</h6>
      <div class="form-check mb-2">
        <input class="form-check-input" type="checkbox" name="use_global" id="use_global" <?= $message['use_global'] ? 'checked' : '' ?>>
        <label class="form-check-label" for="use_global">Usar config global do painel</label>
      </div>

      <div id="overrideArea" style="<?= $message['use_global'] ? 'display:none;' : '' ?>">
        <div class="mb-2">
          <label class="form-label">Efeito de transição</label>
          <select name="transition_effect" class="form-select form-select-sm">
            <?php foreach ($effects as $k=>$label): ?>
              <option value="<?= $k ?>" <?= $message['transition_effect']===$k?'selected':'' ?>><?= $label ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="mb-2">
          <label class="form-label">Tempo na tela (5-15s)</label>
          <input type="number" class="form-control form-control-sm" name="duration_sec" min="5" max="15" value="<?= (int)$message['duration_sec'] ?>">
        </div>
        <div class="form-check mb-2">
          <input class="form-check-input" type="checkbox" name="border_animated" id="border_animated" <?= $message['border_animated'] ? 'checked' : '' ?>>
          <label class="form-check-label" for="border_animated">Borda animada</label>
        </div>
        <div class="mb-2">
          <label class="form-label">Velocidade da borda (1-10)</label>
          <input type="number" class="form-control form-control-sm" name="border_speed" min="1" max="10" value="<?= (int)$message['border_speed'] ?>">
        </div>
      </div>

      <hr>
      <h6>Linhas da mensagem</h6>
      <div id="linesContainer">
        <?php foreach ($lines as $line): ?>
        <div class="border rounded p-2 mb-2 line-block">
          <div class="mb-1">
            <label class="form-label">Texto</label>
            <input type="text" name="line_text[]" class="form-control form-control-sm" value="<?= htmlspecialchars($line['text']) ?>">
          </div>
          <div class="row g-1">
            <div class="col-4">
              <label class="form-label">Tamanho</label>
              <input type="number" name="line_size[]" class="form-control form-control-sm" min="6" max="48" value="<?= (int)$line['size'] ?>">
            </div>
            <div class="col-4">
              <label class="form-label">Posição Y (0-95)</label>
              <input type="number" name="line_y[]" class="form-control form-control-sm" min="0" max="95" value="<?= (int)$line['y'] ?>">
            </div>
            <div class="col-4">
              <label class="form-label">Offset X (-96 a 96)</label>
              <input type="number" name="line_offset[]" class="form-control form-control-sm" min="-96" max="96" value="<?= (int)$line['offset'] ?>">
            </div>
          </div>
          <div class="row g-1 mt-1">
            <div class="col-6">
              <label class="form-label">Alinhamento</label>
              <select name="line_align[]" class="form-select form-select-sm">
                <option value="left" <?= $line['align']==='left'?'selected':'' ?>>Esquerda</option>
                <option value="center" <?= $line['align']==='center'?'selected':'' ?>>Centro</option>
                <option value="right" <?= $line['align']==='right'?'selected':'' ?>>Direita</option>
              </select>
            </div>
            <div class="col-6">
              <label class="form-label">Scroll</label>
              <div class="form-check">
                <input class="form-check-input" type="checkbox" name="line_scroll[]" value="1" <?= $line['scroll'] ? 'checked' : '' ?>>
                <span class="form-check-label">Ativar scroll</span>
              </div>
              <input type="number" name="line_scroll_speed[]" class="form-control form-control-sm mt-1" min="1" max="20" value="<?= (int)$line['scroll_speed'] ?>">
            </div>
          </div>
          <button type="button" class="btn btn-outline-danger btn-sm mt-2 btn-remove-line">Remover linha</button>
        </div>
        <?php endforeach; ?>
      </div>

      <button type="button" class="btn btn-secondary btn-sm" id="addLine">+ Adicionar linha</button>

      <button type="submit" class="btn btn-primary btn-sm mt-3">Salvar</button>
    </form>
  </div>

  <div class="col-md-7">
    <h6>Preview 96x96</h6>
    <canvas id="previewCanvas" width="480" height="480" style="background:#000; border:1px solid #333;"></canvas>
    <p class="mt-2"><small>Pré-visualização aproximada. O desenho 1:1 será definido na API / firmware.</small></p>
  </div>
</div>

<script>
const useGlobal = document.getElementById('use_global');
useGlobal.addEventListener('change', () => {
  document.getElementById('overrideArea').style.display = useGlobal.checked ? 'none' : 'block';
});

const linesContainer = document.getElementById('linesContainer');
document.getElementById('addLine').addEventListener('click', () => {
  const tpl = document.querySelector('.line-block');
  if (!tpl) return;
  const clone = tpl.cloneNode(true);
  clone.querySelectorAll('input').forEach(inp => {
    if (inp.name.startsWith('line_text')) inp.value = '';
  });
  linesContainer.appendChild(clone);
  bindRemoveButtons();
});

function bindRemoveButtons() {
  document.querySelectorAll('.btn-remove-line').forEach(btn => {
    btn.onclick = () => {
      const block = btn.closest('.line-block');
      if (document.querySelectorAll('.line-block').length > 1) {
        block.remove();
      }
    };
  });
}
bindRemoveButtons();

// preview básico
const canvas = document.getElementById('previewCanvas');
const ctx = canvas.getContext('2d');
const LED_COLS = 96;
const LED_ROWS = 96;
const scale = canvas.width / LED_COLS;

function drawCircle(x, y, on) {
  const cx = x * scale + scale/2;
  const cy = y * scale + scale/2;
  const r = scale * 0.38;
  ctx.beginPath();
  ctx.arc(cx, cy, r, 0, Math.PI * 2);
  ctx.fillStyle = on ? '#f8f8f8' : '#202020';
  ctx.fill();
}

function clearMatrix() {
  ctx.fillStyle = '#000';
  ctx.fillRect(0,0,canvas.width, canvas.height);
  for (let y=0; y<LED_ROWS; y++) {
    for (let x=0; x<LED_COLS; x++) {
      drawCircle(x,y,false);
    }
  }
}

let tick = 0;

let t0 = performance.now();
function collectLines(){
  const blocks = document.querySelectorAll('.line-block');
  const lines = [];
  blocks.forEach((b,idx)=>{
    const text = b.querySelector('input[name^="line_text"]')?.value || '';
    const size = parseInt(b.querySelector('input[name^="line_size"]')?.value || '18',10);
    const align = b.querySelector('select[name^="line_align"]')?.value || 'center';
    const x = parseInt(b.querySelector('input[name^="line_x"]')?.value || '48',10);
    const y = parseInt(b.querySelector('input[name^="line_y"]')?.value || '48',10);
    lines.push({text,size,align,x,y});
  });
  return lines;
}
function loop(){
  requestAnimationFrame(loop);
  if(!window.LED_SIM) return;
  const t = (performance.now()-t0)/1000;
  const mtx = window.__msgMatrix || (window.__msgMatrix = LED_SIM.makeMatrix());
  LED_SIM.clear(mtx);
  const msg = {lines_json: JSON.stringify(collectLines())};
  const cfg = {fontFamily: (document.querySelector('[name="font_family"]')?.value)||'Arial', thinPasses:1, threshold:205};
  LED_SIM.renderMessage(mtx, msg, cfg, t);
  if(document.getElementById('border_animated')?.checked){
    LED_SIM.drawBorder(mtx, t, parseInt(document.getElementById('border_speed')?.value||'5',10));
  }
  LED_SIM.drawToCanvas(mtx, canvas);
}
loop();

</script>
<script src="assets/led_sim.js"></script>

<?php require 'footer.php'; ?>
