<?php
header("Content-Type: application/json; charset=utf-8");
date_default_timezone_set("America/Sao_Paulo");
require_once __DIR__ . "/_db.php";

function get_json_input() {
  $raw = file_get_contents("php://input");
  if (!$raw) return null;
  $j = json_decode($raw, true);
  return is_array($j) ? $j : null;
}

$in = get_json_input();
if (!$in) $in = $_POST;

$panel_key  = isset($in["panel_key"]) ? trim($in["panel_key"]) : "";
if ($panel_key === "") { http_response_code(400); echo json_encode(["ok"=>false,"error"=>"panel_key required"]); exit; }

$fw_version = isset($in["fw_version"]) ? trim($in["fw_version"]) : null;
$mac        = isset($in["mac"]) ? trim($in["mac"]) : null;
$ip_local   = isset($in["ip_local"]) ? trim($in["ip_local"]) : null;
$rssi       = isset($in["rssi"]) ? intval($in["rssi"]) : null;
$uptime_ms  = isset($in["uptime_ms"]) ? intval($in["uptime_ms"]) : null;
$free_heap  = isset($in["free_heap"]) ? intval($in["free_heap"]) : null;
$free_psram = isset($in["free_psram"]) ? intval($in["free_psram"]) : null;
$wifi_ssid  = isset($in["wifi_ssid"]) ? trim($in["wifi_ssid"]) : null;
$status     = isset($in["status"]) ? trim($in["status"]) : "ok";
$last_error = isset($in["last_error"]) ? trim($in["last_error"]) : null;
$payload_hash = isset($in["payload_hash"]) ? trim($in["payload_hash"]) : null;
$temp_c     = isset($in["temp_c"]) ? floatval($in["temp_c"]) : null;
$brightness = isset($in["brightness"]) ? intval($in["brightness"]) : null;
$mode       = isset($in["mode"]) ? trim($in["mode"]) : "playlist";
$sig        = isset($in["sig"]) ? trim($in["sig"]) : null;
// Logs (monitor/serial)
$logs_in = null;
if (isset($in["logs"])) $logs_in = $in["logs"];
else if (isset($in["log"])) $logs_in = $in["log"];
$log_level_default = isset($in["log_level"]) ? trim($in["log_level"]) : "info";
// Optional command results (array of {id,status,result})
$cmd_results = isset($in["cmd_results"]) ? $in["cmd_results"] : null;

function gen_secret() {
  // 32 bytes -> 64 hex chars (fácil de colar no firmware)
  try {
    return bin2hex(random_bytes(32));
  } catch (Exception $e) {
    return bin2hex(openssl_random_pseudo_bytes(32));
  }
}

$pdo->beginTransaction();
try {
  $st = $pdo->prepare("SELECT id, secret FROM ic_panels WHERE panel_key=? LIMIT 1");
  $st->execute([$panel_key]);
  $row = $st->fetch();
  $panel_id = null;
  $secret = null;

  if (!$row) {
    $secret_new = gen_secret();
    $ins = $pdo->prepare("INSERT INTO ic_panels(panel_key,secret,is_active) VALUES(?,?,1)");
    $ins->execute([$panel_key,$secret_new]);
    $panel_id = intval($pdo->lastInsertId());
    $secret = $secret_new;
  } else {
    $panel_id = intval($row["id"]);
    $secret = $row["secret"];
  }

  // Assinatura (sig) é opcional enquanto atualizamos o firmware.
  // Se vier, validamos. Se não vier e existir secret, aceitamos e marcamos como "sig_missing".
  $auth_note = null;
  if ($secret !== null && $secret !== "") {
    if ($sig) {
      $base = $panel_key . "|" . strval($fw_version) . "|" . strval($uptime_ms) . "|" . $secret;
      $calc = hash("sha256", $base);
      if (!hash_equals($calc, $sig)) {
        // Log invalid signature before rejecting (keep lightweight)
        try {
          $insChkFail = $pdo->prepare("INSERT INTO ic_panel_logs(panel_id,panel_key,level,message) VALUES(?,?,?,?)");
          $insChkFail->execute([$panel_id, $panel_key, "chk", "sig_invalid"]);
        } catch (Exception $e) { /* ignore */ }
        $pdo->rollBack();
        http_response_code(403);
        echo json_encode(["ok"=>false,"error"=>"invalid signature"]);
        exit;
      }
      $auth_note = "sig_ok";
    } else {
      $auth_note = "sig_missing";
    }
  }

  // Se já veio um status, preserva. Caso contrário, usa o status de autenticação.
  if ((!$status || $status === "ok") && $auth_note === "sig_missing") {
    $status = "sig_missing";
  }

  if ($auth_note && !$last_error) {
    $last_error = $auth_note;
  }

  $ins2 = $pdo->prepare("
    INSERT INTO ic_panel_checkins
      (panel_id,panel_key,fw_version,mac,ip_local,rssi,uptime_ms,free_heap,free_psram,wifi_ssid,status,last_error,payload_hash,temp_c,brightness,mode)
    VALUES
      (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
  ");
  $ins2->execute([
    $panel_id,$panel_key,$fw_version,$mac,$ip_local,$rssi,$uptime_ms,$free_heap,$free_psram,$wifi_ssid,$status,$last_error,$payload_hash,$temp_c,$brightness,$mode
  ]);

  // Log check-ins (throttle to avoid heavy volume)
  $should_log_chk = true;
  $stLastChk = $pdo->prepare("SELECT created_at FROM ic_panel_logs WHERE panel_id=? AND level='chk' ORDER BY id DESC LIMIT 1");
  $stLastChk->execute([$panel_id]);
  $lastChk = $stLastChk->fetch();
  if ($lastChk) {
    $lastChkTs = strtotime($lastChk["created_at"]);
    if ($lastChkTs && (time() - $lastChkTs) < 150) { // ~5 check-ins at 30s
      $should_log_chk = false;
    }
  }
  if ($should_log_chk) {
    $insChk = $pdo->prepare("INSERT INTO ic_panel_logs(panel_id,panel_key,level,message) VALUES(?,?,?,?)");
    $msg = "checkin ok";
    if ($uptime_ms !== null) $msg .= " uptime_ms=" . $uptime_ms;
    if ($status) $msg .= " status=" . $status;
    $insChk->execute([$panel_id, $panel_key, "chk", $msg]);
  }

  // Save logs if provided
  if ($logs_in) {
    if (is_string($logs_in)) $logs_in = [$logs_in];
    if (is_array($logs_in)) {
      $insLog = $pdo->prepare("INSERT INTO ic_panel_logs(panel_id,panel_key,level,message) VALUES(?,?,?,?)");
      foreach ($logs_in as $entry) {
        $level = $log_level_default;
        $msg = null;
        if (is_array($entry)) {
          if (isset($entry["level"])) $level = trim($entry["level"]);
          if (isset($entry["message"])) $msg = (string)$entry["message"];
          else if (isset($entry["msg"])) $msg = (string)$entry["msg"];
        } else {
          $msg = (string)$entry;
        }
        if ($msg === null || $msg === "") continue;
        $insLog->execute([$panel_id,$panel_key,$level ?: "info",$msg]);
      }
    }
  }

  // Optional: command result updates
  if (is_array($cmd_results)) {
    $updCmd = $pdo->prepare("UPDATE ic_panel_commands SET status=?, result=?, done_at=NOW() WHERE id=? AND panel_id=?");
    foreach ($cmd_results as $cr) {
      if (!is_array($cr)) continue;
      $cid = isset($cr["id"]) ? intval($cr["id"]) : 0;
      if ($cid <= 0) continue;
      $cstatus = isset($cr["status"]) ? trim($cr["status"]) : "done";
      $cresult = isset($cr["result"]) ? trim($cr["result"]) : null;
      $updCmd->execute([$cstatus ?: "done", $cresult, $cid, $panel_id]);
    }
  }

  $pdo->commit();
  echo json_encode(["ok"=>true,"panel_key"=>$panel_key,"saved"=>true]);
} catch (Exception $e) {
  $pdo->rollBack();
  http_response_code(500);
  echo json_encode(["ok"=>false,"error"=>$e->getMessage()]);
}
?>
