<?php
require_once __DIR__ . "/auth.php";
require_login();
require_role(["admin_master","admin_cliente"]);

require_once __DIR__ . "/../_db.php";

// ---------------- Helpers ----------------
function norm_key($s) {
  $s = strtolower(trim((string)$s));
  $s = preg_replace('/\s+/', '_', $s);
  $s = str_replace(['-','.'], '_', $s);
  return $s;
}

function parse_bool01($v, $default=1) {
  if ($v === null || $v === "") return $default;
  $s = strtolower(trim((string)$v));
  if ($s === "1" || $s === "true" || $s === "sim" || $s === "yes" || $s === "y") return 1;
  if ($s === "0" || $s === "false" || $s === "nao" || $s === "não" || $s === "no" || $s === "n") return 0;
  return $default;
}

function parse_price_to_cents($v) {
  if ($v === null) return null;
  $s = trim((string)$v);
  if ($s === "") return null;
  // Accept: 10,50 | 10.50 | 10 | R$ 10,50
  $s = preg_replace('/[^0-9,\.]/', '', $s);
  if ($s === "") return null;
  // If contains both , and . assume thousand/decimal -> keep last as decimal
  if (strpos($s, ',') !== false && strpos($s, '.') !== false) {
    $lastComma = strrpos($s, ',');
    $lastDot   = strrpos($s, '.');
    $decPos = max($lastComma, $lastDot);
    $intPart = preg_replace('/[^0-9]/', '', substr($s, 0, $decPos));
    $decPart = preg_replace('/[^0-9]/', '', substr($s, $decPos + 1));
    $decPart = substr($decPart . "00", 0, 2);
    if ($intPart === "") $intPart = "0";
    return intval($intPart) * 100 + intval($decPart);
  }
  if (strpos($s, ',') !== false) {
    $s = str_replace('.', '', $s); // treat dot as thousand
    $s = str_replace(',', '.', $s);
  }
  $f = floatval($s);
  return intval(round($f * 100));
}

function parse_duration_sec($v, $default=4) {
  if ($v === null || $v === "") return $default;
  $n = intval($v);
  if ($n <= 0) $n = $default;
  if ($n > 120) $n = 120;
  return $n;
}

function csv_read_rows($path, &$errors) {
  $rows = [];
  $h = fopen($path, 'r');
  if (!$h) {
    $errors[] = "Não foi possível abrir o arquivo.";
    return $rows;
  }
  $header = null;
  while (($data = fgetcsv($h, 0, ',')) !== false) {
    if ($header === null) {
      // If delimiter is ; try again
      $header = $data;
      if (count($header) <= 1) {
        rewind($h);
        $header = null;
        break;
      }
      continue;
    }
    $rows[] = $data;
  }
  fclose($h);

  if ($header !== null) {
    return [$header, $rows];
  }

  // Retry with ';'
  $rows = [];
  $h = fopen($path, 'r');
  $header = null;
  while (($data = fgetcsv($h, 0, ';')) !== false) {
    if ($header === null) {
      $header = $data;
      continue;
    }
    $rows[] = $data;
  }
  fclose($h);
  return [$header ?: [], $rows];
}

// ---------------- Context / client selection ----------------
$me = current_user();
$is_master = ($me && $me['role'] === 'admin_master');

$client_id = 0;
if ($is_master) {
  $client_id = isset($_GET['client_id']) ? intval($_GET['client_id']) : 0;
  if (isset($_POST['client_id'])) $client_id = intval($_POST['client_id']);
} else {
  $client_id = intval($me['client_id']);
}

// Load clients for dropdown (master only)
$clients = [];
if ($is_master) {
  $st = $pdo->query("SELECT id, name FROM clients ORDER BY name");
  $clients = $st->fetchAll();
}

$messages = [];
$errors = [];
$inserted = 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['do_import'])) {
  if ($client_id <= 0) {
    $errors[] = "Selecione um cliente.";
  }

  $mode = isset($_POST['mode']) ? $_POST['mode'] : 'file';
  $tmpPath = null;

  if (!$errors) {
    if ($mode === 'url') {
      $url = trim((string)($_POST['csv_url'] ?? ''));
      if ($url === '') {
        $errors[] = "Cole a URL CSV (Google Sheets publicado como CSV).";
      } else {
        $ctx = stream_context_create(['http' => ['timeout' => 15]]);
        $csv = @file_get_contents($url, false, $ctx);
        if ($csv === false || trim($csv) === '') {
          $errors[] = "Não consegui baixar o CSV pela URL.";
        } else {
          $tmpPath = tempnam(sys_get_temp_dir(), 'ic_csv_');
          file_put_contents($tmpPath, $csv);
        }
      }
    } else {
      if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        $errors[] = "Envie um arquivo .csv.";
      } else {
        $tmpPath = $_FILES['csv_file']['tmp_name'];
      }
    }
  }

  if (!$errors && $tmpPath) {
    list($header, $rows) = csv_read_rows($tmpPath, $errors);
    if (!$errors) {
      $cols = [];
      foreach ($header as $i => $hname) $cols[$i] = norm_key($hname);

      $allowed = [
        'name','header','title','subtitle','subtitle_position','price_cents','price','unit','aux','footer',
        'duration_sec','enabled','opt_bold','opt_invert','opt_blink','opt_big','opt_center'
      ];
      $allowed = array_flip($allowed);

      $pdo->beginTransaction();
      try {
        $sql = "INSERT INTO offers
          (client_id,name,header,title,subtitle,subtitle_position,price_cents,unit,aux,footer,duration_sec,enabled,opt_bold,opt_invert,opt_blink,opt_big,opt_center)
          VALUES
          (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
        $ins = $pdo->prepare($sql);

        foreach ($rows as $rIdx => $r) {
          $row = [];
          foreach ($cols as $i => $key) {
            if (!isset($allowed[$key])) continue;
            $row[$key] = isset($r[$i]) ? $r[$i] : "";
          }

          $name = trim((string)($row['name'] ?? ''));
          $title = trim((string)($row['title'] ?? ''));
          if ($name === '' && $title === '') {
            // skip empty line
            continue;
          }

          $subtitle_pos = trim((string)($row['subtitle_position'] ?? 'bottom'));
          if ($subtitle_pos !== 'top' && $subtitle_pos !== 'bottom') $subtitle_pos = 'bottom';

          $price_cents = null;
          if (isset($row['price_cents']) && trim((string)$row['price_cents']) !== '') {
            $price_cents = intval($row['price_cents']);
          } elseif (isset($row['price']) && trim((string)$row['price']) !== '') {
            $price_cents = parse_price_to_cents($row['price']);
          }

          $duration = parse_duration_sec($row['duration_sec'] ?? null, 4);
          $enabled = parse_bool01($row['enabled'] ?? null, 1);

          $opt_bold   = parse_bool01($row['opt_bold'] ?? null, 0);
          $opt_invert = parse_bool01($row['opt_invert'] ?? null, 0);
          $opt_blink  = parse_bool01($row['opt_blink'] ?? null, 0);
          $opt_big    = parse_bool01($row['opt_big'] ?? null, 0);
          $opt_center = parse_bool01($row['opt_center'] ?? null, 1);

          $ins->execute([
            $client_id,
            $name,
            trim((string)($row['header'] ?? '')),
            $title,
            trim((string)($row['subtitle'] ?? '')),
            $subtitle_pos,
            $price_cents,
            trim((string)($row['unit'] ?? '')),
            trim((string)($row['aux'] ?? '')),
            trim((string)($row['footer'] ?? '')),
            $duration,
            $enabled,
            $opt_bold,$opt_invert,$opt_blink,$opt_big,$opt_center
          ]);
          $inserted++;
        }

        $pdo->commit();
        $messages[] = "Importação concluída: {$inserted} oferta(s) inserida(s).";
      } catch (Exception $e) {
        $pdo->rollBack();
        $errors[] = "Erro ao importar: " . $e->getMessage();
      }
    }
  }
}

require_once __DIR__ . "/header.php";
?>

<div class="container" style="max-width: 980px;">
  <div class="d-flex align-items-center justify-content-between mt-4 mb-3">
    <h2 class="m-0">Importar Ofertas (CSV)</h2>
    <a class="btn btn-outline-secondary" href="offers_list.php<?= $client_id?('?client_id='.$client_id):'' ?>">Voltar</a>
  </div>

  <?php if ($errors): ?>
    <div class="alert alert-danger">
      <strong>Falhou:</strong><br>
      <?= implode("<br>", array_map('htmlspecialchars', $errors)) ?>
    </div>
  <?php endif; ?>

  <?php if ($messages): ?>
    <div class="alert alert-success">
      <?= implode("<br>", array_map('htmlspecialchars', $messages)) ?>
    </div>
  <?php endif; ?>

  <div class="alert alert-info">
    <div><strong>Dica:</strong> para Excel/Google Sheets, exporte como <b>CSV</b>.</div>
    <div class="mt-2">Colunas aceitas (cabeçalho): <code>name,header,title,subtitle,subtitle_position,price,price_cents,unit,aux,footer,duration_sec,enabled,opt_bold,opt_invert,opt_blink,opt_big,opt_center</code></div>
    <div class="mt-1">Ex.: <code>price</code> pode ser <code>10,50</code> ou <code>10.50</code> (vira centavos automaticamente).</div>
  </div>

  <form method="post" enctype="multipart/form-data" class="card p-3">
    <?php if ($is_master): ?>
      <div class="mb-3">
        <label class="form-label">Cliente</label>
        <select class="form-select" name="client_id" required>
          <option value="">Selecione...</option>
          <?php foreach ($clients as $c): ?>
            <option value="<?= intval($c['id']) ?>" <?= ($client_id==intval($c['id']))?'selected':'' ?>><?= htmlspecialchars($c['name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
    <?php endif; ?>

    <div class="mb-3">
      <label class="form-label">Origem</label>
      <div class="form-check">
        <input class="form-check-input" type="radio" name="mode" value="file" id="mode_file" checked>
        <label class="form-check-label" for="mode_file">Enviar arquivo CSV</label>
      </div>
      <div class="form-check">
        <input class="form-check-input" type="radio" name="mode" value="url" id="mode_url">
        <label class="form-check-label" for="mode_url">URL CSV (Google Sheets publicado)</label>
      </div>
    </div>

    <div class="mb-3" id="box_file">
      <label class="form-label">Arquivo .csv</label>
      <input class="form-control" type="file" name="csv_file" accept=".csv,text/csv">
    </div>

    <div class="mb-3" id="box_url" style="display:none;">
      <label class="form-label">URL CSV</label>
      <input class="form-control" type="url" name="csv_url" placeholder="https://docs.google.com/spreadsheets/.../pub?output=csv">
      <small class="text-muted">No Google Sheets: Arquivo → Compartilhar → Publicar na Web → CSV.</small>
    </div>

    <input type="hidden" name="do_import" value="1">
    <button class="btn btn-primary">Importar</button>
  </form>

  <div class="card p-3 mt-3">
    <div class="mb-2"><strong>Modelo CSV (copie e cole):</strong></div>
    <pre style="white-space: pre-wrap; margin:0;">name,header,title,subtitle,subtitle_position,price,unit,aux,footer,duration_sec,enabled,opt_bold,opt_invert,opt_blink,opt_big,opt_center
Picanha,OFERTA,PICANHA,KG,bottom,39,90,KG,,Somente hoje!,4,1,1,0,0,1,1
</pre>
  </div>
</div>

<script>
  const modeFile = document.getElementById('mode_file');
  const modeUrl  = document.getElementById('mode_url');
  const boxFile  = document.getElementById('box_file');
  const boxUrl   = document.getElementById('box_url');
  function syncMode(){
    if (modeUrl.checked) {
      boxUrl.style.display = '';
      boxFile.style.display = 'none';
    } else {
      boxFile.style.display = '';
      boxUrl.style.display = 'none';
    }
  }
  modeFile.addEventListener('change', syncMode);
  modeUrl.addEventListener('change', syncMode);
  syncMode();
</script>

<?php require_once __DIR__ . "/footer.php"; ?>
