// led_sim_atlas_loader.js
// Compatível com JSONs que usam meta.map ou root glyphs.
// Mantém cache/promise, suporte base /iCartaz, notFound guard e fallback.
(function(){
  if(!window.LED_SIM) window.LED_SIM = {};
  LED_SIM._atlases = LED_SIM._atlases || {};

  function detectBase(){
    if(window._ICARTAZ_BASE) return String(window._ICARTAZ_BASE).replace(/\/+$/,'');
    try {
      if(location && location.pathname && location.pathname.indexOf('/iCartaz/') !== -1) return '/iCartaz';
    } catch(e){}
    return '';
  }
  const BASE = detectBase();

  // Internal loader that returns a Promise
  LED_SIM._loadAtlasInternal = function(lines){
    const key = String(lines);
    LED_SIM._atlases[key] = LED_SIM._atlases[key] || {};

    // If previously determined missing, skip
    if(LED_SIM._atlases[key].notFound){
      return Promise.reject(new Error('atlas not found (cached)'));
    }
    // reuse promise if already loading
    if (LED_SIM._atlases[key].promise) return LED_SIM._atlases[key].promise;
    // if already ready return resolved promise
    if (LED_SIM._atlases[key].ready) return Promise.resolve(LED_SIM._atlases[key]);

    const jsonUrl = (BASE ? (BASE + '/admin/fonts/') : '/admin/fonts/') + 'impact_atlas_' + lines + '.json';
    const imgUrl  = (BASE ? (BASE + '/admin/fonts/') : '/admin/fonts/') + 'impact_atlas_' + lines + '.png';

    const p = fetch(jsonUrl, { cache:'no-cache' }).then(r=>{
      if(!r.ok){
        if(r.status === 404) LED_SIM._atlases[key].notFound = true;
        throw new Error('atlas json not found: ' + r.status);
      }
      return r.json();
    }).then(json=>{
      // Normalização: suportar JSONs que têm "glyphs" no root ou em meta
      const meta = json.meta || {};
      // Prefer meta.map, senão procurar json.glyphs ou json.map ou meta.glyphs
      const possibleMap = meta.map || json.map || json.glyphs || meta.glyphs || null;
      if(possibleMap && typeof possibleMap === 'object' && Object.keys(possibleMap).length > 0){
        meta.map = possibleMap;
      }
      // If after normalization meta.map is missing or empty, treat as notFound
      if(!meta || !meta.map || typeof meta.map !== 'object' || Object.keys(meta.map).length === 0){
        LED_SIM._atlases[key].notFound = true;
        throw new Error('atlas json has empty/invalid map');
      }

      // load image
      return new Promise((resolve, reject)=>{
        const img = new Image();
        img.crossOrigin = 'anonymous';
        img.onload = function(){
          LED_SIM._atlases[key].meta = meta;
          LED_SIM._atlases[key].img = img;
          LED_SIM._atlases[key].ready = true;
          LED_SIM._atlases[key].error = false;
          delete LED_SIM._atlases[key].promise;
          resolve(LED_SIM._atlases[key]);
        };
        img.onerror = function(e){
          LED_SIM._atlases[key].error = true;
          LED_SIM._atlases[key].ready = false;
          LED_SIM._atlases[key].notFound = true;
          delete LED_SIM._atlases[key].promise;
          reject(new Error('atlas image load error'));
        };
        img.src = imgUrl + '?_=' + Date.now();
      });
    }).catch(err=>{
      LED_SIM._atlases[key].error = true;
      LED_SIM._atlases[key].ready = false;
      LED_SIM._atlases[key].lastFailedAt = Date.now();
      delete LED_SIM._atlases[key].promise;
      throw err;
    });

    LED_SIM._atlases[key].promise = p;
    return p;
  };

  // Public: loadAtlas(lines, cb)
  LED_SIM.loadAtlas = function(lines, cb){
    const key = String(lines);
    // if marked notFound, callback with error immediately
    if(LED_SIM._atlases[key] && LED_SIM._atlases[key].notFound){
      if(cb) cb(new Error('atlas not found (cached)'));
      return;
    }
    if (LED_SIM._atlases[key] && LED_SIM._atlases[key].ready){
      if(cb) cb(null, LED_SIM._atlases[key]);
      return;
    }
    LED_SIM._loadAtlasInternal(lines).then(atlas=>{
      if(cb) cb(null, atlas);
    }).catch(err=>{
      if(cb) cb(err);
    });
  };

  LED_SIM.loadAtlasPromise = function(lines){
    return LED_SIM._loadAtlasInternal(lines);
  };

  // drawTextFromAtlas: tolerant, returns false when atlas not available or produced no pixels
  LED_SIM.drawTextFromAtlas = function(m, text, box, lines, cfg){
    const key = String(lines);
    const atlas = LED_SIM._atlases[key];
    if(!atlas || !atlas.ready || !atlas.meta || !atlas.img) return false;

    const meta = atlas.meta;
    const img = atlas.img;
    const map = meta.map || {};
    const letter_spacing = (cfg && typeof cfg.letter_spacing === 'number') ? cfg.letter_spacing : (meta.letter_spacing || 1);
    const threshold = (cfg && typeof cfg.threshold === 'number') ? cfg.threshold : 170;

    // build glyphs list and advance
    let totalAdvance = 0;
    const glyphs = [];
    for(const ch of Array.from(text)){
      const g = map[ch] || map[ch.toUpperCase()] || map[' '] || { x:0,y:0,w:Math.round(meta.px*0.6),h:Math.round(meta.px), xadvance: Math.round(meta.px*0.6) };
      glyphs.push(g);
      totalAdvance += (g.xadvance || g.w) + letter_spacing;
    }

    const availW = box.x1 - box.x0 + 1;
    const scale = Math.min(1, availW / Math.max(1, totalAdvance));
    const glyphHeight = meta.px;
    const boxH = box.y1 - box.y0 + 1;
    const topOffset = Math.floor((boxH - Math.round(glyphHeight * scale)) / 2);
    const baselineY = box.y0 + topOffset + Math.round(meta.px * 0.8 * scale);

    const tmp = document.createElement('canvas');
    tmp.width = 96;
    tmp.height = 96;
    const tctx = tmp.getContext('2d');
    tctx.fillStyle = '#000';
    tctx.fillRect(0,0,tmp.width,tmp.height);
    tctx.imageSmoothingEnabled = true;
    tctx.imageSmoothingQuality = 'high';

    let xPos = Math.floor((box.x0 + box.x1)/2 - Math.round(totalAdvance*scale)/2);
    let anyGlyphDrawn = false;

    for(const g of glyphs){
      const sx = g.x, sy = g.y, sw = g.w, sh = g.h;
      const dw = Math.max(1, Math.round(sw * scale));
      const dh = Math.max(1, Math.round(sh * scale));
      try {
        if(!sw || !sh) { xPos += Math.round((g.xadvance || g.w) * scale) + Math.round(letter_spacing * scale); continue; }
        tctx.drawImage(img, sx, sy, sw, sh, Math.round(xPos), Math.round(baselineY - dh + Math.round(meta.px*0.2*scale)), dw, dh);
        anyGlyphDrawn = true;
      } catch(e){
        // ignore drawing errors
      }
      xPos += Math.round((g.xadvance || g.w) * scale) + Math.round(letter_spacing * scale);
    }

    if(!anyGlyphDrawn) return false;

    const imgData = tctx.getImageData(0,0,tmp.width,tmp.height).data;
    let hits = 0;
    for(let yy=0; yy<tmp.height; yy++){
      for(let xx=0; xx<tmp.width; xx++){
        const idx = (yy*tmp.width + xx)*4;
        const a = imgData[idx+3];
        const r = imgData[idx];
        const val = (a>0)? r : 0;
        if(val >= threshold){
          if(xx >= box.x0 && xx <= box.x1 && yy >= box.y0 && yy <= box.y1){
            LED_SIM.setPixel(m, xx, yy, 1);
            hits++;
          }
        }
      }
    }

    if(hits === 0) return false;
    return true;
  };

})();