// admin/assets/led_sim.js
// Firmware-accurate LED renderer (96x96) using FLB bitmap fonts
(function(){
  const LED_COLS = 96, LED_ROWS = 96;

  function makeMatrix(){ const m = new Uint8Array(LED_COLS*LED_ROWS); m.fill(0); return m; }
  function clear(m){ m.fill(0); }
  function setPixel(m,x,y,on){ if(x<0||y<0||x>=LED_COLS||y>=LED_ROWS) return; m[y*LED_COLS+x]= on?1:0; }
  function getPixel(m,x,y){ if(x<0||y<0||x>=LED_COLS||y>=LED_ROWS) return 0; return m[y*LED_COLS+x]; }

  function normalizeText(s){
    if(!s) return '';
    let out = '';
    for(let i=0;i<s.length;i++){
      let c = s.charCodeAt(i);
      // keep within Latin-1 range, approximate firmware behavior
      if(c > 0xFF){ out += ' '; continue; }
      // uppercase ASCII
      if(c >= 97 && c <= 122) c = c - 32;
      switch(c){
        case 0xC0: case 0xC1: case 0xC2: case 0xC3: case 0xC4: case 0xC5: c = 65; break; // A
        case 0xC7: c = 67; break; // C
        case 0xC8: case 0xC9: case 0xCA: case 0xCB: c = 69; break; // E
        case 0xCC: case 0xCD: case 0xCE: case 0xCF: c = 73; break; // I
        case 0xD1: c = 78; break; // N
        case 0xD2: case 0xD3: case 0xD4: case 0xD5: case 0xD6: c = 79; break; // O
        case 0xD9: case 0xDA: case 0xDB: case 0xDC: c = 85; break; // U
        case 0xDD: case 0xDE: c = 89; break; // Y
        case 0xE0: case 0xE1: case 0xE2: case 0xE3: case 0xE4: case 0xE5: c = 65; break;
        case 0xE7: c = 67; break;
        case 0xE8: case 0xE9: case 0xEA: case 0xEB: c = 69; break;
        case 0xEC: case 0xED: case 0xEE: case 0xEF: c = 73; break;
        case 0xF1: c = 78; break;
        case 0xF2: case 0xF3: case 0xF4: case 0xF5: case 0xF6: c = 79; break;
        case 0xF9: case 0xFA: case 0xFB: case 0xFC: c = 85; break;
        case 0xFD: case 0xFF: c = 89; break;
      }
      out += String.fromCharCode(c);
    }
    return out;
  }

  function getFont(name){
    if(!window.FLB_FONTS || !window.FLB_FONTS[name]) return null;
    return window.FLB_FONTS[name];
  }

  function flb_char_index(ch){
    const code = ch.charCodeAt(0);
    if(code < 32 || code > 127) return 0;
    return code - 32;
  }

  function flb_vis_w(f){ return f.width; }
  function flb_vis_h(f){ return f.height; }

  function flb_glyph_width(f, idx){
    if(idx < 0 || idx >= 96) return f.width;
    const w = f.glyph_widths[idx];
    return w === 0 ? f.width : w;
  }

  function flb_spacing_for_font(f){
    if(f.height >= 60) return 3;
    if(f.height >= 42) return 3;
    if(f.height >= 28) return 2;
    if(f.height >= 17) return 2;
    return 1;
  }

  function flb_spacing_value(f, idx, spacing_override, spacing_adjust){
    if(spacing_override >= 0) return spacing_override;
    return spacing_adjust;
  }

  function flb_text_width(f, s, spacing_override){
    if(!s || !s.length) return 0;
    let w = 0;
    const spacing_adjust = flb_spacing_for_font(f);
    for(let i=0;i<s.length;i++){
      const idx = flb_char_index(s[i]);
      w += flb_glyph_width(f, idx);
      if(i + 1 < s.length){
        w += flb_spacing_value(f, idx, spacing_override, spacing_adjust);
      }
    }
    return w;
  }

  function flb_draw_glyph(m, f, x, y, idx){
    if(idx < 0 || idx >= 96) idx = 0;
    const bpr = f.glyph_bpr[idx];
    const off = f.glyph_offsets[idx];
    if(!bpr) return;
    const glyph_w = flb_glyph_width(f, idx);
    const data = f.data;
    for(let row=0; row<f.height; row++){
      let col = 0;
      for(let t=0; t<bpr; t++){
        const decode = data[off + row*bpr + t];
        for(let bit=0; bit<8 && col < glyph_w; bit++){
          if(decode & (1 << bit)) setPixel(m, x + col, y + row, 1);
          col++;
        }
      }
    }
  }

  function flb_draw_text(m, x, y, f, s, spacing_override){
    let cx = x;
    const spacing_adjust = flb_spacing_for_font(f);
    for(let i=0;i<s.length;i++){
      const idx = flb_char_index(s[i]);
      flb_draw_glyph(m, f, cx, y, idx);
      if(i + 1 < s.length){
        cx += flb_glyph_width(f, idx) + flb_spacing_value(f, idx, spacing_override, spacing_adjust);
      } else {
        cx += flb_glyph_width(f, idx);
      }
    }
  }

  function flb_draw_text_in_box(m, x, y, w, h, f, s, spacing, elapsed_ms, x_offset, y_offset, allow_scroll){
    if(!s || !s.length) return;
    const text_w = flb_text_width(f, s, spacing);
    let draw_x = x + Math.floor((w - text_w) / 2) + x_offset;
    const draw_y = y + Math.floor((h - flb_vis_h(f)) / 2) + y_offset;
    if(allow_scroll && text_w > w){
      const speed = 8;
      const gap = 16;
      const dash_w = 4;
      const gap2 = 16;
      const sep_w = gap + dash_w + gap2;
      const cycle_w = text_w + sep_w;
      const base_x = x + x_offset;
      const shift = cycle_w > 0 ? Math.floor((elapsed_ms * speed) / 1000) % cycle_w : 0;
      draw_x = base_x - shift;
      const dash_y = draw_y + Math.floor((flb_vis_h(f) - 1) / 2);
      for(let k=0; k<3; k++){
        const seg_x = draw_x + k * cycle_w;
        flb_draw_text(m, seg_x, draw_y, f, s, spacing);
        const dash_x = seg_x + text_w + gap;
        for(let dx=0; dx<dash_w; dx++) setPixel(m, dash_x + dx, dash_y, 1);
      }
      return;
    }
    flb_draw_text(m, draw_x, draw_y, f, s, spacing);
  }

  function select_font_by_size(size){
    const f60 = getFont('FLB_60X35');
    const f42 = getFont('FLB_42X24');
    const f28 = getFont('FLB_28X16');
    const f17 = getFont('FLB_17X8');
    const f7  = getFont('FLB_7X4');
    const n = parseInt(size || 0, 10);
    const exact = {
      60: f60,
      42: f42,
      28: f28,
      17: f17,
      7:  f7,
    };
    if(exact[n]) return exact[n];
    if(n >= 60 && f60) return f60;
    if(n >= 42 && f42) return f42;
    if(n >= 28 && f28) return f28;
    if(n >= 17 && f17) return f17;
    return f7 || f17 || f28 || f42 || f60;
  }

  function text_width_auto(s, size){
    const f = select_font_by_size(size);
    return flb_text_width(f, s, -1);
  }

  function draw_text_auto(m, x, y, s, size){
    const f = select_font_by_size(size);
    flb_draw_text(m, Math.floor(x), Math.floor(y), f, s, -1);
  }

  function draw_price_centered(m, price_cents, unit, blink, elapsed_ms, box_y, x_offset, y_offset, measure_only){
    let show_price = true;
    if(blink){
      const blink_phase_ms = 120;
      const blink_total = blink_phase_ms * 6;
      if(elapsed_ms < blink_total){
        show_price = (Math.floor(elapsed_ms / blink_phase_ms) % 2) === 0;
      }
    }

    const reais = Math.floor(price_cents / 100);
    const cents = Math.abs(price_cents % 100);
    const reais_s = String(reais);
    const real_digits = reais_s.length;
    const cents_digits = (cents < 10) ? ('0' + String(cents)) : String(cents);
    const comma_s = ',';
    const unit_s = normalizeText(unit || '');

    const field_x = 4;
    const field_w = 88;
    const gap_reais_comma = 1;
    const gap_comma_cents = 1;

    const f42 = getFont('FLB_42X24');
    const f28 = getFont('FLB_28X16');
    const f17 = getFont('FLB_17X8');
    const f7  = getFont('FLB_7X4');
    const fComma = getFont('FLB_VIRGULA');

    if(!f42 && !f28 && !f17 && !f7) return 0;

    const allow28 = real_digits >= 3;
    let font_reais = (allow28 && f28) ? f28 : (f42 || f28 || f17 || f7);
    if(!font_reais) font_reais = f17 || f7 || f28 || f42;
    if(!font_reais) return 0;
    const font_comma = fComma || f7 || f17 || f28 || f42;
    const font_unit = f7 || f17 || f28 || f42;
    if(!font_comma || !font_unit) return 0;
    let font_cents = f28 || f17 || f7;

    let price_h = 0;
    let cents_h = 0;
    let unit_h = 0;
    const line_h = 1;
    const min_gap = 2;
    let gap1 = min_gap;
    let gap2 = min_gap;

    const clamp_right_gaps = (price_h_val) => {
      let total = cents_h + gap1 + line_h + gap2 + unit_h;
      while(total > price_h_val && (gap1 > min_gap || gap2 > min_gap)){
        if(gap1 >= gap2 && gap1 > min_gap) gap1--;
        else if(gap2 > min_gap) gap2--;
        total = cents_h + gap1 + line_h + gap2 + unit_h;
      }
      return total;
    };
    let total_right_h = 0;

    const spacing_reais = 2;
    const spacing_cents = 2;
    const spacing_unit = 1;
    let reais_w = 0;
    let cents_w = 0;
    let comma_w = 0;
    let unit_w = 0;
    let right_w = 0;
    let total_w = 0;

    const stepDownFont = (f) => {
      if(f === f42 && allow28 && f28) return f28;
      if(f === f28 && f17) return f17;
      return f17 || f7 || f28 || f42;
    };

    // Step down font size until the price fits the field width.
    while(true){
      if(font_reais === f17) font_cents = f7 || f17;
      else if(font_reais === f28) font_cents = f17 || f7;
      else font_cents = f28 || f17 || f7;
      price_h = flb_vis_h(font_reais);
      cents_h = flb_vis_h(font_cents);
      unit_h = flb_vis_h(font_unit);
      gap1 = min_gap;
      gap2 = min_gap;
      total_right_h = clamp_right_gaps(price_h);
      if(total_right_h < price_h){
        const extra = price_h - total_right_h;
        const add1 = Math.floor(extra / 2);
        const add2 = extra - add1;
        gap1 += add1;
        gap2 += add2;
        total_right_h = price_h;
      }

      reais_w = flb_text_width(font_reais, reais_s, spacing_reais);
      cents_w = flb_text_width(font_cents, cents_digits, spacing_cents);
      comma_w = flb_text_width(font_comma, comma_s, -1);
      unit_w = flb_text_width(font_unit, unit_s, spacing_unit);
      right_w = Math.max(cents_w, unit_w);
      total_w = reais_w + gap_reais_comma + comma_w + gap_comma_cents + right_w;

      if(total_w <= field_w || font_reais === f17 || !allow28) break;
      font_reais = stepDownFont(font_reais);
    }

    const block_h = (total_right_h > price_h) ? total_right_h : price_h;
    const start_x = field_x + Math.floor((field_w - total_w) / 2) + x_offset;
    const reais_x = start_x;
    const reais_y = box_y + Math.floor((block_h - price_h) / 2) + y_offset;
    if(!measure_only && show_price) flb_draw_text(m, reais_x, reais_y, font_reais, reais_s, spacing_reais);

    const comma_x = reais_x + reais_w + gap_reais_comma;
    const right_x = comma_x + comma_w + gap_comma_cents;
    const right_top = box_y + Math.floor((block_h - total_right_h) / 2) + y_offset;
    const cents_y = right_top;
    const comma_y = cents_y + cents_h - flb_vis_h(font_comma) + 3;
    if(!measure_only && show_price){
      flb_draw_text(m, comma_x, comma_y, font_comma, comma_s, -1);
      const cents_x = right_x + Math.floor((right_w - cents_w) / 2);
      flb_draw_text(m, cents_x, cents_y, font_cents, cents_digits, spacing_cents);
    }

    const line_y = cents_y + cents_h + gap1;
    if(!measure_only && show_price){
      for(let x=right_x; x<right_x+right_w; x++) setPixel(m, x, line_y, 1);
    }

    const unit_y = line_y + line_h + gap2;
    if(!measure_only && show_price){
      const unit_x = right_x + Math.floor((right_w - unit_w) / 2);
      flb_draw_text(m, unit_x, unit_y, font_unit, unit_s, spacing_unit);
    }
    return block_h;
  }

  function draw_border_dashed_chamfer(m, phase, clear_first){
    if(clear_first) clear(m);
    const dashOn = 3;
    const dashOff = 1;
    const dashLen = dashOn + dashOff;
    const r = 3;
    const w = LED_COLS;
    const h = LED_ROWS;

    const draw_if_on = (pos, x, y) => {
      if(((pos + phase) % dashLen) < dashOn) setPixel(m, x, y, 1);
    };

    let pos = 0;
    for(let x=r; x<w-r; x++, pos++) draw_if_on(pos, x, 0);
    for(let i=0; i<r; i++, pos++) draw_if_on(pos, w - r + i, i);
    for(let y=r; y<h-r; y++, pos++) draw_if_on(pos, w - 1, y);
    for(let i=0; i<r; i++, pos++) draw_if_on(pos, w - 1 - i, h - r + i);
    for(let x=w - r - 1; x>=r; x--, pos++) draw_if_on(pos, x, h - 1);
    for(let i=0; i<r; i++, pos++) draw_if_on(pos, r - 1 - i, h - 1 - i);
    for(let y=h - r - 1; y>=r; y--, pos++) draw_if_on(pos, 0, y);
    for(let i=0; i<r; i++, pos++) draw_if_on(pos, i, r - 1 - i);
  }

  function drawBorder(m, tsec, speed){
    const spd = Math.max(1, Math.min(10, speed || 5));
    let interval = 200 - (spd * 15);
    if(interval < 20) interval = 20;
    const elapsed_ms = Math.max(0, Math.floor((tsec || 0) * 1000));
    const phase = Math.floor(elapsed_ms / interval) & 0x3F;
    draw_border_dashed_chamfer(m, phase, false);
  }

  function renderOffer(m, offer, cfg, tsec){
    clear(m);
    const elapsed_ms = Math.max(0, Math.floor((tsec || 0) * 1000));

    const subtitle = normalizeText((offer && offer.subtitle) || '');
    const title = normalizeText((offer && offer.title) || '');
    const desc = normalizeText((offer && offer.description) || '');
    const footer = normalizeText((offer && offer.footer) || '');

    const field_x = 4;
    const field_w = 88;
    const border = 1;
    const gap_sub = 4;
    const gap_title = 3;
    const gap_desc = 5;
    const bottom_gap = 4;
    const sub_h = 7;
    const title_h = 17;
    const desc_h = 7;

    let y = border + gap_sub;
    const sub_y = y;
    y += sub_h + gap_title;
    const title_y = y;
    y += title_h;

    const f7 = getFont('FLB_7X4');
    const f17 = getFont('FLB_17X8');

    if(subtitle.length){
      flb_draw_text_in_box(m, field_x + 1, sub_y, field_w - 2, sub_h, f7, subtitle, 1, elapsed_ms, 0, 0, true);
    }

    if(title.length){
      flb_draw_text_in_box(m, field_x + 1, title_y, field_w - 2, title_h, f17, title, 2, elapsed_ms, 0, 0, true);
    }

    const desc_line = desc.length ? desc : footer;
    const desc_y = LED_ROWS - bottom_gap - border - desc_h;
    const price_h = draw_price_centered(
      m,
      offer && offer.price_cents ? offer.price_cents : 0,
      offer && offer.unit ? offer.unit : '',
      cfg && cfg.price_blink,
      elapsed_ms,
      0,
      0,
      0,
      true
    );
    const price_top = title_y + title_h;
    const price_bottom = desc_y - gap_desc;
    const avail = price_bottom - price_top;
    const center_bias = 3;
    let price_y = price_top + Math.floor((avail - price_h) / 2) + center_bias;
    if(price_y < price_top) price_y = price_top;
    draw_price_centered(
      m,
      offer && offer.price_cents ? offer.price_cents : 0,
      offer && offer.unit ? offer.unit : '',
      cfg && cfg.price_blink,
      elapsed_ms,
      price_y,
      0,
      0,
      false
    );

    if(desc_line.length){
      flb_draw_text_in_box(m, field_x + 1, desc_y, field_w - 2, desc_h, f7, desc_line, 1, elapsed_ms, 0, 0, true);
    }

  }

  function renderMessage(m, message, cfg, tsec){
    clear(m);
    const elapsed_ms = Math.max(0, Math.floor((tsec || 0) * 1000));
    const x_offset = 0;
    const y_offset = 0;
    let lines = [];
    if(message && Array.isArray(message.lines)) lines = message.lines;
    else if(message && message.lines_json){
      try { lines = JSON.parse(message.lines_json); } catch(e){ lines = []; }
    }
    if(!Array.isArray(lines)) lines = [];

    for(let i=0;i<lines.length;i++){
      const l = lines[i] || {};
      const text = normalizeText((l.text || '').toString());
      if(!text) continue;
      const size = parseInt(l.size || 14, 10);
      const w = text_width_auto(text, size);
      let x = parseInt(l.offset || 0, 10) || 0;
      if(l.align === 'center') x = Math.floor((LED_COLS - w) / 2);
      else if(l.align === 'right') x = LED_COLS - w - 1;

      const do_scroll = !!l.scroll || (w > LED_COLS);
      if(do_scroll){
        const speed = Math.max(1, parseInt(l.scroll_speed || 8, 10));
        const gap = 16;
        const dash_w = 4;
        const gap2 = 16;
        const sep_w = gap + dash_w + gap2;
        const cycle_w = w + sep_w;
        const shift = cycle_w > 0 ? Math.floor((elapsed_ms * speed) / 1000) % cycle_w : 0;
        const base_x = x + x_offset;
        const y = (parseInt(l.y || 0, 10) || 0) + y_offset;
        const f = select_font_by_size(size);
        const dash_y = y + Math.floor((flb_vis_h(f) - 1) / 2);
        const start_x = base_x - shift;
        for(let k=0; k<3; k++){
          const seg_x = start_x + k * cycle_w;
          draw_text_auto(m, Math.floor(seg_x), Math.floor(y), text, size);
          const dash_x = seg_x + w + gap;
          for(let dx=0; dx<dash_w; dx++) setPixel(m, dash_x + dx, dash_y, 1);
        }
        continue;
      }

      const y = parseInt(l.y || 0, 10) || 0;
      draw_text_auto(m, Math.floor(x + x_offset), Math.floor(y + y_offset), text, size);
    }

  }

  function drawToCanvas(m, canvas, opts){
    const ctx = canvas.getContext('2d');
    const scale = canvas.width / LED_COLS;
    ctx.fillStyle = '#000';
    ctx.fillRect(0,0,canvas.width,canvas.height);
    const solid = !!(opts && opts.solid);
    for(let y=0;y<LED_ROWS;y++){
      for(let x=0;x<LED_COLS;x++){
        const on = m[y*LED_COLS+x]===1;
        const cx = x*scale + scale/2;
        const cy = y*scale + scale/2;
        ctx.fillStyle = on ? '#f8f8f8' : '#1a1a1a';
        if(solid){
          ctx.fillRect(x*scale, y*scale, scale, scale);
        } else {
          const r = scale*0.38;
          ctx.beginPath();
          ctx.arc(cx, cy, r, 0, Math.PI*2);
          ctx.fill();
        }
      }
    }
  }

  window.LED_SIM = {
    LED_COLS, LED_ROWS,
    makeMatrix, clear, setPixel, getPixel,
    renderOffer, renderMessage, drawBorder, drawToCanvas,
    normalizeText
  };
})();
